<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CommunityPage;
use App\Helpers\ImageHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CommunityPageController extends Controller
{
    /**
     * Display a listing of community pages.
     */
    public function index()
    {
        $pages = CommunityPage::orderBy('page_key')->get();
        return view('admin.community-pages.index', compact('pages'));
    }

    /**
     * Show the form for creating a new community page.
     */
    public function create()
    {
        return view('admin.community-pages.create');
    }

    /**
     * Store a newly created community page.
     */
    public function store(Request $request)
    {
        $request->validate([
            'page_key' => 'required|string|max:255|unique:community_pages,page_key',
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'is_active' => 'boolean'
        ]);

        $page = new CommunityPage($request->except(['images']));
        $page->is_active = $request->boolean('is_active');

        // Handle multiple image uploads
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $filename = 'community_page_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('images/community-pages'), $filename);
                $imagesPaths[] = 'community-pages/' . $filename;
            }
        }
        $page->images = $imagesPaths;

        $page->save();

        return redirect()->route('admin.community-pages.index')
                        ->with('success', 'Community page created successfully.');
    }

    /**
     * Display the specified community page.
     */
    public function show(CommunityPage $communityPage)
    {
        return view('admin.community-pages.show', compact('communityPage'));
    }

    /**
     * Show the form for editing the specified community page.
     */
    public function edit(CommunityPage $communityPage)
    {
        return view('admin.community-pages.edit', compact('communityPage'));
    }

    /**
     * Update the specified community page.
     */
    public function update(Request $request, CommunityPage $communityPage)
    {
        $request->validate([
            'page_key' => 'required|string|max:255|unique:community_pages,page_key,' . $communityPage->id,
            'title' => 'required|string|max:255',
            'content' => 'nullable|string',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'is_active' => 'boolean'
        ]);

        // Handle image uploads
        $existingImages = $communityPage->images ?? [];
        $imagesPaths = $existingImages;

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $filename = 'community_page_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('images/community-pages'), $filename);
                $imagesPaths[] = 'community-pages/' . $filename;
            }
        }

        // Remove selected images if requested
        if ($request->filled('remove_images')) {
            $removeImages = $request->remove_images;
            foreach ($removeImages as $imageToRemove) {
                if (($key = array_search($imageToRemove, $imagesPaths)) !== false) {
                    unset($imagesPaths[$key]);
                    // Delete file from filesystem
                    if (file_exists(public_path('images/' . $imageToRemove))) {
                        unlink(public_path('images/' . $imageToRemove));
                    }
                }
            }
            $imagesPaths = array_values($imagesPaths); // Re-index array
        }

        $communityPage->fill($request->except(['images', 'remove_images']));
        $communityPage->images = $imagesPaths;
        $communityPage->is_active = $request->boolean('is_active');
        $communityPage->save();

        return redirect()->route('admin.community-pages.index')
                        ->with('success', 'Community page updated successfully.');
    }

    /**
     * Remove the specified community page.
     */
    public function destroy(CommunityPage $communityPage)
    {
        // Delete all associated images
        if ($communityPage->images) {
            foreach ($communityPage->images as $imagePath) {
                if (file_exists(public_path('images/' . $imagePath))) {
                    unlink(public_path('images/' . $imagePath));
                }
            }
        }

        $communityPage->delete();

        return redirect()->route('admin.community-pages.index')
                        ->with('success', 'Community page deleted successfully.');
    }

    /**
     * Toggle active status of community page.
     */
    public function toggleStatus(CommunityPage $communityPage)
    {
        $communityPage->is_active = !$communityPage->is_active;
        $communityPage->save();

        $status = $communityPage->is_active ? 'activated' : 'deactivated';
        
        return response()->json([
            'success' => true,
            'message' => "Community page {$status} successfully.",
            'is_active' => $communityPage->is_active
        ]);
    }
}
