<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Order;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    /**
     * Display a listing of customers.
     */
    public function index(Request $request)
    {
        $query = User::withCount(['orders', 'wishlists']);

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by status
        if ($request->has('status') && $request->status != 'all') {
            if ($request->status == 'blocked') {
                $query->where('is_blocked', true);
            } elseif ($request->status == 'active') {
                $query->where('is_blocked', false);
            }
        }

        $customers = $query->latest()->paginate(15);

        return view('admin.customers.index', compact('customers'));
    }

    /**
     * Display the specified customer.
     */
    public function show(User $user)
    {
        $user->load(['orders' => function($query) {
            $query->latest()->take(10);
        }]);

        // Calculate customer statistics
        $stats = [
            'total_orders' => $user->orders()->count(),
            'total_spent' => $user->orders()->where('payment_status', 'paid')->sum('total_amount'),
            'average_order' => $user->orders()->where('payment_status', 'paid')->avg('total_amount'),
            'last_order' => $user->orders()->latest()->first(),
            'wishlist_count' => $user->wishlists()->count(),
        ];

        return view('admin.customers.show', compact('user', 'stats'));
    }

    /**
     * Block a customer.
     */
    public function block(User $user)
    {
        if ($user->is_admin) {
            return back()->with('error', 'Cannot block admin users!');
        }

        $user->update(['is_blocked' => true]);

        return back()->with('success', 'Customer blocked successfully!');
    }

    /**
     * Unblock a customer.
     */
    public function unblock(User $user)
    {
        $user->update(['is_blocked' => false]);

        return back()->with('success', 'Customer unblocked successfully!');
    }
}