<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DiscountCampaign;
use App\Models\Product;
use App\Models\Category;
use App\Models\UserDiscount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DiscountController extends Controller
{
    public function index()
    {
        $campaigns = DiscountCampaign::latest()->paginate(10);
        $userDiscounts = UserDiscount::with('user')->latest()->paginate(10);
        
        return view('admin.discounts.index', compact('campaigns', 'userDiscounts'));
    }

    public function create()
    {
        $products = Product::active()->get();
        $categories = Category::where('is_active', true)->get();
        
        return view('admin.discounts.create', compact('products', 'categories'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_percentage' => 'required|numeric|min:1|max:100',
            'starts_at' => 'required|date',
            'ends_at' => 'required|date|after:starts_at',
            'discount_type' => 'required|in:all,products,categories',
            'product_ids' => 'required_if:discount_type,products|array',
            'category_ids' => 'required_if:discount_type,categories|array',
            'banner_image' => 'nullable|image|max:2048',
            'is_seasonal' => 'nullable|boolean',
            'show_timer' => 'nullable|boolean'
        ]);

        $data = [
            'name' => $validated['name'],
            'description' => $validated['description'],
            'discount_percentage' => $validated['discount_percentage'],
            'starts_at' => $validated['starts_at'],
            'ends_at' => $validated['ends_at'],
            'is_active' => true,
            'is_seasonal' => $validated['is_seasonal'] ?? false,
            'show_timer' => $validated['show_timer'] ?? false
        ];

        // Handle discount type
        if ($validated['discount_type'] === 'products') {
            $data['product_ids'] = $validated['product_ids'];
        } elseif ($validated['discount_type'] === 'categories') {
            $data['category_ids'] = $validated['category_ids'];
        }

        // Handle banner image
        if ($request->hasFile('banner_image')) {
            $data['banner_image'] = $request->file('banner_image')->store('discounts', 'public');
        }

        $campaign = DiscountCampaign::create($data);

        // Handle product-based sales with pivot table
        if ($validated['discount_type'] === 'products' && !empty($validated['product_ids'])) {
            $products = Product::whereIn('id', $validated['product_ids'])->get();
            $pivotData = [];

            foreach ($products as $product) {
                $originalPrice = $product->price;
                $discountAmount = ($originalPrice * $validated['discount_percentage']) / 100;
                $salePrice = $originalPrice - $discountAmount;

                $pivotData[$product->id] = [
                    'original_price' => $originalPrice,
                    'sale_price' => $salePrice,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }

            $campaign->products()->attach($pivotData);
        }

        return redirect()->route('admin.discounts.index')
            ->with('success', 'Sale campaign created successfully! Products have been assigned with discounted prices.');
    }

    public function edit(DiscountCampaign $discount)
    {
        $products = Product::active()->get();
        $categories = Category::where('is_active', true)->get();
        
        return view('admin.discounts.edit', compact('discount', 'products', 'categories'));
    }

    public function update(Request $request, DiscountCampaign $discount)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'discount_percentage' => 'required|numeric|min:1|max:100',
            'starts_at' => 'required|date',
            'ends_at' => 'required|date|after:starts_at',
            'discount_type' => 'required|in:all,products,categories',
            'product_ids' => 'required_if:discount_type,products|array',
            'category_ids' => 'required_if:discount_type,categories|array',
            'banner_image' => 'nullable|image|max:2048'
        ]);

        $data = [
            'name' => $validated['name'],
            'description' => $validated['description'],
            'discount_percentage' => $validated['discount_percentage'],
            'starts_at' => $validated['starts_at'],
            'ends_at' => $validated['ends_at']
        ];

        // Handle discount type
        if ($validated['discount_type'] === 'all') {
            $data['product_ids'] = null;
            $data['category_ids'] = null;
        } elseif ($validated['discount_type'] === 'products') {
            $data['product_ids'] = $validated['product_ids'];
            $data['category_ids'] = null;
        } elseif ($validated['discount_type'] === 'categories') {
            $data['product_ids'] = null;
            $data['category_ids'] = $validated['category_ids'];
        }

        // Handle banner image
        if ($request->hasFile('banner_image')) {
            // Delete old image
            if ($discount->banner_image) {
                Storage::disk('public')->delete($discount->banner_image);
            }
            $data['banner_image'] = $request->file('banner_image')->store('discounts', 'public');
        }

        $discount->update($data);

        return redirect()->route('admin.discounts.index')
            ->with('success', 'Discount campaign updated successfully!');
    }

    public function destroy(DiscountCampaign $discount)
    {
        if ($discount->banner_image) {
            Storage::disk('public')->delete($discount->banner_image);
        }
        
        $discount->delete();
        
        return redirect()->route('admin.discounts.index')
            ->with('success', 'Discount campaign deleted successfully!');
    }

    public function toggleActive(DiscountCampaign $discount)
    {
        $discount->update(['is_active' => !$discount->is_active]);
        
        return back()->with('success', 'Campaign status updated successfully!');
    }
}