<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\InstagramFeed;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class InstagramController extends Controller
{
    /**
     * Display a listing of Instagram posts.
     */
    public function index()
    {
        $feeds = InstagramFeed::orderBy('sort_order')
                              ->orderBy('created_at', 'desc')
                              ->paginate(12);
        
        $sizeRecommendation = '1080x1080 pixels (Square format)';
        
        return view('admin.instagram.index', compact('feeds', 'sizeRecommendation'));
    }

    /**
     * Show the form for creating a new Instagram post.
     */
    public function create()
    {
        $sizeRecommendation = '1080x1080 pixels (Square format)';
        
        return view('admin.instagram.create', compact('sizeRecommendation'));
    }

    /**
     * Store a newly created Instagram post.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
            'link' => 'required|url',
            'caption' => 'nullable|string|max:500',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean'
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('instagram', 'public');
            $validated['image'] = $path;
        }

        $validated['is_active'] = $request->has('is_active');
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        InstagramFeed::create($validated);

        return redirect()->route('admin.instagram.index')
                        ->with('success', 'Instagram post added successfully!');
    }

    /**
     * Show the form for editing the specified Instagram post.
     */
    public function edit(InstagramFeed $instagram)
    {
        $sizeRecommendation = '1080x1080 pixels (Square format)';
        $feed = $instagram; // Alias for view compatibility
        
        return view('admin.instagram.edit', compact('feed', 'sizeRecommendation'));
    }

    /**
     * Update the specified Instagram post.
     */
    public function update(Request $request, InstagramFeed $instagram)
    {
        $validated = $request->validate([
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'link' => 'required|url',
            'caption' => 'nullable|string|max:500',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean'
        ]);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($instagram->image && Storage::disk('public')->exists($instagram->image)) {
                Storage::disk('public')->delete($instagram->image);
            }
            
            $path = $request->file('image')->store('instagram', 'public');
            $validated['image'] = $path;
        }

        $validated['is_active'] = $request->has('is_active');
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        $instagram->update($validated);

        return redirect()->route('admin.instagram.index')
                        ->with('success', 'Instagram post updated successfully!');
    }

    /**
     * Remove the specified Instagram post.
     */
    public function destroy(InstagramFeed $instagram)
    {
        // Delete image file
        if ($instagram->image && Storage::disk('public')->exists($instagram->image)) {
            Storage::disk('public')->delete($instagram->image);
        }

        $instagram->delete();

        return redirect()->route('admin.instagram.index')
                        ->with('success', 'Instagram post deleted successfully!');
    }

    /**
     * Toggle active status.
     */
    public function toggleActive(InstagramFeed $instagram)
    {
        $instagram->update(['is_active' => !$instagram->is_active]);

        return redirect()->route('admin.instagram.index')
                        ->with('success', 'Instagram post status updated!');
    }
}