<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use PDF;

class OrderController extends Controller
{
    /**
     * Display a listing of orders.
     */
    public function index(Request $request)
    {
        $query = Order::with(['user', 'items.product']);

        // Filter by status
        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }

        // Filter by date range
        if ($request->has('from_date') && $request->from_date) {
            $query->whereDate('created_at', '>=', $request->from_date);
        }
        if ($request->has('to_date') && $request->to_date) {
            $query->whereDate('created_at', '<=', $request->to_date);
        }

        // Search by order number or customer
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                  ->orWhere('customer_email', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%");
            });
        }

        $orders = $query->latest()->paginate(15)->withQueryString();
        
        return view('admin.orders.index', compact('orders'));
    }

    /**
     * Display the specified order.
     */
    public function show(Order $order)
    {
        $order->load(['items.product', 'user', 'shippingAddress']);
        return view('admin.orders.show', compact('order'));
    }

    /**
     * Update order status.
     */
    public function updateStatus(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => 'required|in:pending,processing,shipped,out_for_delivery,delivered,cancelled,refunded'
        ]);

        $oldStatus = $order->status;
        
        // Update status
        $order->status = $validated['status'];
        
        // Set timestamps based on status
        if ($validated['status'] == 'shipped' && !$order->shipped_at) {
            $order->shipped_at = now();
        } elseif ($validated['status'] == 'delivered' && !$order->delivered_at) {
            $order->delivered_at = now();
        }
        
        $order->save();
        
        // Add tracking update
        $order->addTrackingUpdate(
            "Order status changed from " . ucfirst($oldStatus) . " to " . ucfirst($validated['status']),
            $validated['status']
        );

        // Send email notification (if Mail class exists)
        try {
            if (class_exists('App\Mail\OrderStatusUpdate')) {
                Mail::to($order->customer_email)->send(new \App\Mail\OrderStatusUpdate($order, $oldStatus));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send order status email: ' . $e->getMessage());
        }

        // Return JSON response for AJAX requests
        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Order status updated successfully']);
        }

        return back()->with('success', 'Order status updated successfully');
    }

    /**
     * Update tracking information.
     */
    public function updateTracking(Request $request, Order $order)
    {
        $validated = $request->validate([
            'tracking_number' => 'nullable|string|max:255',
            'carrier' => 'nullable|string|max:50',
            'estimated_delivery' => 'nullable|date',
            'update_message' => 'required|string|max:500'
        ]);
        
        // Update tracking fields
        if (isset($validated['tracking_number'])) {
            $order->tracking_number = $validated['tracking_number'];
        }
        if (isset($validated['carrier'])) {
            $order->carrier = $validated['carrier'];
        }
        if (isset($validated['estimated_delivery'])) {
            $order->estimated_delivery = $validated['estimated_delivery'];
        }
        
        // Add tracking update to history
        $order->addTrackingUpdate($validated['update_message']);
        
        // Save the order
        $order->save();
        
        // Send tracking update email (if configured)
        try {
            if ($order->tracking_number && class_exists('App\Mail\TrackingUpdate')) {
                Mail::to($order->customer_email)->send(new \App\Mail\TrackingUpdate($order));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send tracking update email: ' . $e->getMessage());
        }
        
        return back()->with('success', 'Tracking information updated successfully');
    }

    /**
     * Mark order as shipped.
     */
    public function markAsShipped(Request $request, Order $order)
    {
        $validated = $request->validate([
            'tracking_number' => 'required|string|max:255',
            'carrier' => 'required|string|max:50',
            'estimated_delivery' => 'required|date'
        ]);

        $order->update([
            'status' => 'shipped',
            'tracking_number' => $validated['tracking_number'],
            'carrier' => $validated['carrier'],
            'estimated_delivery' => $validated['estimated_delivery'],
            'shipped_at' => now()
        ]);

        // Add tracking update
        $order->addTrackingUpdate(
            'Order has been shipped with ' . strtoupper($validated['carrier']) . '. Tracking: ' . $validated['tracking_number'],
            'shipped'
        );

        // Send shipping notification email
        try {
            if (class_exists('App\Mail\OrderShipped')) {
                Mail::to($order->customer_email)->send(new \App\Mail\OrderShipped($order));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send shipping notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Order marked as shipped successfully');
    }

    /**
     * Cancel an order.
     */
    public function cancel(Request $request, Order $order)
    {
        // Check if order can be cancelled
        if (!$order->canBeCancelled()) {
            return back()->with('error', 'This order cannot be cancelled');
        }

        $order->update(['status' => 'cancelled']);
        
        // Add tracking update
        $order->addTrackingUpdate('Order has been cancelled', 'cancelled');

        // Restore product stock
        foreach ($order->items as $item) {
            if ($item->product) {
                $item->product->increment('stock', $item->quantity);
            }
        }

        // Send cancellation email
        try {
            if (class_exists('App\Mail\OrderCancelled')) {
                Mail::to($order->customer_email)->send(new \App\Mail\OrderCancelled($order));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send cancellation email: ' . $e->getMessage());
        }

        return back()->with('success', 'Order cancelled successfully');
    }

    /**
     * Generate invoice for an order.
     */
    public function invoice(Order $order)
    {
        $order->load(['items.product', 'shippingAddress', 'user']);
        
        // Get company info for invoice
        $companyInfo = \App\Models\CompanyInfo::first();

        // Check if PDF library is available
        if (!class_exists('PDF')) {
            // Return HTML view if PDF not available
            return view('admin.orders.invoice', compact('order', 'companyInfo'));
        }

        // Generate PDF invoice
        $pdf = PDF::loadView('admin.orders.invoice', compact('order', 'companyInfo'));
        
        return $pdf->download('invoice-' . $order->order_number . '.pdf');
    }

    /**
     * Process refund for an order.
     */
    public function refund(Request $request, Order $order)
    {
        // Check if order can be refunded
        if (!$order->canBeRefunded()) {
            return back()->with('error', 'This order cannot be refunded');
        }

        $validated = $request->validate([
            'refund_amount' => 'required|numeric|min:0|max:' . $order->total_amount,
            'refund_reason' => 'required|string|max:500'
        ]);

        // Update order status
        $order->update([
            'status' => 'refunded',
            'payment_status' => 'refunded'
        ]);
        
        // Add tracking update
        $order->addTrackingUpdate(
            'Refund processed: $' . number_format($validated['refund_amount'], 2) . '. Reason: ' . $validated['refund_reason'],
            'refunded'
        );

        // Process refund through payment gateway (implement based on your payment method)
        // This is a placeholder - implement actual refund logic based on payment provider
        try {
            // Example for Stripe refund
            if ($order->payment_method == 'stripe' && $order->transaction_id) {
                // \Stripe\Refund::create([
                //     'charge' => $order->transaction_id,
                //     'amount' => $validated['refund_amount'] * 100, // Convert to cents
                // ]);
            }
        } catch (\Exception $e) {
            Log::error('Refund processing failed: ' . $e->getMessage());
            return back()->with('error', 'Refund processing failed. Please process manually.');
        }

        // Send refund confirmation email
        try {
            if (class_exists('App\Mail\OrderRefunded')) {
                Mail::to($order->customer_email)->send(new \App\Mail\OrderRefunded($order, $validated['refund_amount']));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send refund email: ' . $e->getMessage());
        }

        return back()->with('success', 'Order refunded successfully');
    }
}