<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Display reports dashboard.
     */
    public function index()
    {
        // Get date range
        $startDate = Carbon::now()->startOfMonth();
        $endDate = Carbon::now()->endOfMonth();

        // Sales overview
        $salesData = [
            'total_revenue' => Order::whereBetween('created_at', [$startDate, $endDate])
                ->where('payment_status', 'paid')
                ->sum('total_amount'),
            'total_orders' => Order::whereBetween('created_at', [$startDate, $endDate])->count(),
            'average_order' => Order::whereBetween('created_at', [$startDate, $endDate])
                ->where('payment_status', 'paid')
                ->avg('total_amount'),
            'new_customers' => User::whereBetween('created_at', [$startDate, $endDate])->count(),
        ];

        // Sales by day for chart
        $salesByDay = Order::where('payment_status', 'paid')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, SUM(total_amount) as total, COUNT(*) as orders')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Top products
        $topProducts = Product::withCount(['orderItems as sold_count' => function($query) use ($startDate, $endDate) {
                $query->whereHas('order', function($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate])
                      ->where('payment_status', 'paid');
                });
            }])
            ->orderBy('sold_count', 'desc')
            ->take(5)
            ->get();

        // Top categories
        $topCategories = Category::withCount(['products as revenue' => function($query) use ($startDate, $endDate) {
                $query->join('order_items', 'products.id', '=', 'order_items.product_id')
                    ->join('orders', 'order_items.order_id', '=', 'orders.id')
                    ->whereBetween('orders.created_at', [$startDate, $endDate])
                    ->where('orders.payment_status', 'paid')
                    ->select(DB::raw('SUM(order_items.total)'));
            }])
            ->orderBy('revenue', 'desc')
            ->take(5)
            ->get();

        return view('admin.reports.index', compact(
            'salesData', 'salesByDay', 'topProducts', 'topCategories', 'startDate', 'endDate'
        ));
    }

    /**
     * Sales report.
     */
    public function sales(Request $request)
    {
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : Carbon::now()->startOfMonth();
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : Carbon::now()->endOfMonth();

        $orders = Order::with(['user', 'items'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->where('payment_status', 'paid')
            ->latest()
            ->paginate(20);

        $totals = [
            'revenue' => $orders->sum('total_amount'),
            'orders' => $orders->count(),
            'average' => $orders->avg('total_amount'),
        ];

        return view('admin.reports.sales', compact('orders', 'totals', 'startDate', 'endDate'));
    }

    /**
     * Products report.
     */
    public function products(Request $request)
    {
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : Carbon::now()->startOfMonth();
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : Carbon::now()->endOfMonth();

        $products = Product::with('category')
            ->withCount(['orderItems as units_sold' => function($query) use ($startDate, $endDate) {
                $query->whereHas('order', function($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate])
                      ->where('payment_status', 'paid');
                })->select(DB::raw('SUM(quantity)'));
            }])
            ->withSum(['orderItems as revenue' => function($query) use ($startDate, $endDate) {
                $query->whereHas('order', function($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate])
                      ->where('payment_status', 'paid');
                });
            }], 'total')
            ->orderBy('units_sold', 'desc')
            ->paginate(20);

        return view('admin.reports.products', compact('products', 'startDate', 'endDate'));
    }

    /**
     * Customers report.
     */
    public function customers(Request $request)
    {
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : Carbon::now()->startOfMonth();
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : Carbon::now()->endOfMonth();

        $customers = User::withCount(['orders' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }])
            ->withSum(['orders as total_spent' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate])
                      ->where('payment_status', 'paid');
            }], 'total_amount')
            ->having('orders_count', '>', 0)
            ->orderBy('total_spent', 'desc')
            ->paginate(20);

        return view('admin.reports.customers', compact('customers', 'startDate', 'endDate'));
    }

    /**
     * Export report data.
     */
    public function export($type, Request $request)
    {
        $startDate = $request->start_date ? Carbon::parse($request->start_date) : Carbon::now()->startOfMonth();
        $endDate = $request->end_date ? Carbon::parse($request->end_date) : Carbon::now()->endOfMonth();

        switch ($type) {
            case 'orders':
                $data = Order::with(['user', 'items.product'])
                    ->whereBetween('created_at', [$startDate, $endDate])
                    ->get();
                $filename = 'orders_' . $startDate->format('Y-m-d') . '_to_' . $endDate->format('Y-m-d') . '.csv';
                break;
            
            case 'products':
                $data = Product::with('category')->get();
                $filename = 'products_' . now()->format('Y-m-d') . '.csv';
                break;
            
            case 'customers':
                $data = User::withCount('orders')
                    ->withSum('orders', 'total_amount')
                    ->get();
                $filename = 'customers_' . now()->format('Y-m-d') . '.csv';
                break;
            
            default:
                return back()->with('error', 'Invalid export type');
        }

        // Generate CSV
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($data, $type) {
            $file = fopen('php://output', 'w');

            // Add headers based on type
            if ($type == 'orders') {
                fputcsv($file, ['Order #', 'Date', 'Customer', 'Email', 'Total', 'Status', 'Payment Status']);
                foreach ($data as $order) {
                    fputcsv($file, [
                        $order->order_number,
                        $order->created_at->format('Y-m-d H:i'),
                        $order->customer_name,
                        $order->customer_email,
                        $order->total_amount,
                        $order->status,
                        $order->payment_status
                    ]);
                }
            } elseif ($type == 'products') {
                fputcsv($file, ['SKU', 'Name', 'Category', 'Price', 'Stock', 'Status']);
                foreach ($data as $product) {
                    fputcsv($file, [
                        $product->sku,
                        $product->name,
                        $product->category->name ?? 'N/A',
                        $product->price,
                        $product->stock,
                        $product->is_active ? 'Active' : 'Inactive'
                    ]);
                }
            } elseif ($type == 'customers') {
                fputcsv($file, ['Name', 'Email', 'Total Orders', 'Total Spent', 'Joined']);
                foreach ($data as $customer) {
                    fputcsv($file, [
                        $customer->name,
                        $customer->email,
                        $customer->orders_count,
                        $customer->orders_sum_total_amount ?? 0,
                        $customer->created_at->format('Y-m-d')
                    ]);
                }
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}