<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;

class SettingsController extends Controller
{
    /**
     * Display settings page.
     */
    public function index()
    {
        $settings = [
            'site_name' => config('app.name'),
            'site_url' => config('app.url'),
            'admin_email' => env('MAIL_FROM_ADDRESS', ''),
            'timezone' => config('app.timezone'),
            'currency' => env('CURRENCY_CODE', 'USD'),
            'currency_symbol' => env('CURRENCY_SYMBOL', '$'),
            'tax_rate' => env('TAX_RATE', 10),
            'shipping_rate' => env('SHIPPING_RATE', 10),
            'free_shipping_threshold' => env('FREE_SHIPPING_THRESHOLD', 100),
            'maintenance_mode' => app()->isDownForMaintenance(),
            'allow_guest_checkout' => env('ALLOW_GUEST_CHECKOUT', true),
            'enable_reviews' => env('ENABLE_REVIEWS', true),
            'enable_wishlist' => env('ENABLE_WISHLIST', true),
            'products_per_page' => env('PRODUCTS_PER_PAGE', 12),
            'low_stock_threshold' => env('LOW_STOCK_THRESHOLD', 5),
            'enable_coupons' => env('ENABLE_COUPONS', true),
            'smtp_host' => env('MAIL_HOST', ''),
            'smtp_port' => env('MAIL_PORT', ''),
            'smtp_username' => env('MAIL_USERNAME', ''),
            'smtp_encryption' => env('MAIL_ENCRYPTION', ''),
            'stripe_key' => env('STRIPE_KEY', ''),
            'stripe_webhook_secret' => env('STRIPE_WEBHOOK_SECRET', ''),
            'paypal_client_id' => env('PAYPAL_CLIENT_ID', ''),
            'google_analytics_id' => env('GOOGLE_ANALYTICS_ID', ''),
            'facebook_pixel_id' => env('FACEBOOK_PIXEL_ID', ''),
        ];

        return view('admin.settings.index', compact('settings'));
    }

    /**
     * Update settings.
     */
    public function update(Request $request)
    {
        $validated = $request->validate([
            'site_name' => 'required|string|max:255',
            'admin_email' => 'required|email',
            'currency' => 'required|string|max:3',
            'currency_symbol' => 'required|string|max:5',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'shipping_rate' => 'required|numeric|min:0',
            'free_shipping_threshold' => 'required|numeric|min:0',
            'products_per_page' => 'required|integer|min:1|max:100',
            'low_stock_threshold' => 'required|integer|min:1',
        ]);

        // Update .env file
        $this->updateEnvFile([
            'APP_NAME' => '"' . $validated['site_name'] . '"',
            'MAIL_FROM_ADDRESS' => $validated['admin_email'],
            'CURRENCY_CODE' => $validated['currency'],
            'CURRENCY_SYMBOL' => $validated['currency_symbol'],
            'TAX_RATE' => $validated['tax_rate'],
            'SHIPPING_RATE' => $validated['shipping_rate'],
            'FREE_SHIPPING_THRESHOLD' => $validated['free_shipping_threshold'],
            'PRODUCTS_PER_PAGE' => $validated['products_per_page'],
            'LOW_STOCK_THRESHOLD' => $validated['low_stock_threshold'],
            'ALLOW_GUEST_CHECKOUT' => $request->has('allow_guest_checkout') ? 'true' : 'false',
            'ENABLE_REVIEWS' => $request->has('enable_reviews') ? 'true' : 'false',
            'ENABLE_WISHLIST' => $request->has('enable_wishlist') ? 'true' : 'false',
            'ENABLE_COUPONS' => $request->has('enable_coupons') ? 'true' : 'false',
        ]);

        // Update optional settings if provided
        if ($request->filled('smtp_host')) {
            $this->updateEnvFile([
                'MAIL_HOST' => $request->smtp_host,
                'MAIL_PORT' => $request->smtp_port,
                'MAIL_USERNAME' => $request->smtp_username,
                'MAIL_PASSWORD' => $request->smtp_password ?: env('MAIL_PASSWORD'),
                'MAIL_ENCRYPTION' => $request->smtp_encryption,
            ]);
        }

        if ($request->filled('stripe_key')) {
            $this->updateEnvFile([
                'STRIPE_KEY' => $request->stripe_key,
                'STRIPE_SECRET' => $request->stripe_secret ?: env('STRIPE_SECRET'),
                'STRIPE_WEBHOOK_SECRET' => $request->stripe_webhook_secret,
            ]);
        }

        if ($request->filled('paypal_client_id')) {
            $this->updateEnvFile([
                'PAYPAL_CLIENT_ID' => $request->paypal_client_id,
                'PAYPAL_SECRET' => $request->paypal_secret ?: env('PAYPAL_SECRET'),
            ]);
        }

        if ($request->filled('google_analytics_id')) {
            $this->updateEnvFile(['GOOGLE_ANALYTICS_ID' => $request->google_analytics_id]);
        }

        if ($request->filled('facebook_pixel_id')) {
            $this->updateEnvFile(['FACEBOOK_PIXEL_ID' => $request->facebook_pixel_id]);
        }

        // Clear config cache
        Artisan::call('config:clear');
        Artisan::call('cache:clear');

        // Handle maintenance mode
        if ($request->has('maintenance_mode') && !app()->isDownForMaintenance()) {
            Artisan::call('down', [
                '--message' => 'The site is currently under maintenance. Please check back soon.',
                '--retry' => 3600
            ]);
        } elseif (!$request->has('maintenance_mode') && app()->isDownForMaintenance()) {
            Artisan::call('up');
        }

        return back()->with('success', 'Settings updated successfully! Some changes may require a page refresh to take effect.');
    }

    /**
     * Update .env file values.
     */
    private function updateEnvFile(array $data)
    {
        $envPath = base_path('.env');
        $envContent = file_get_contents($envPath);

        foreach ($data as $key => $value) {
            $pattern = "/^{$key}=.*/m";
            
            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, "{$key}={$value}", $envContent);
            } else {
                $envContent .= "\n{$key}={$value}";
            }
        }

        file_put_contents($envPath, $envContent);
    }

    /**
     * Clear cache.
     */
    public function clearCache()
    {
        Artisan::call('cache:clear');
        Artisan::call('config:clear');
        Artisan::call('route:clear');
        Artisan::call('view:clear');
        
        return back()->with('success', 'All cache cleared successfully!');
    }

    /**
     * Optimize application.
     */
    public function optimize()
    {
        Artisan::call('optimize');
        Artisan::call('config:cache');
        Artisan::call('route:cache');
        Artisan::call('view:cache');
        
        return back()->with('success', 'Application optimized successfully!');
    }
}