<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ShippingZone;
use Illuminate\Http\Request;

class ShippingZoneController extends Controller
{
    /**
     * Display a listing of shipping zones
     */
    public function index()
    {
        $zones = ShippingZone::orderBy('state')
            ->orderBy('city')
            ->paginate(20);
        
        return view('admin.shipping-zones.index', compact('zones'));
    }

    /**
     * Show the form for creating a new shipping zone
     */
    public function create()
    {
        $states = ShippingZone::getStates();
        return view('admin.shipping-zones.create', compact('states'));
    }

    /**
     * Store a newly created shipping zone
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:10',
            'postcode_range' => 'nullable|string|max:50',
            'standard_rate' => 'required|numeric|min:0',
            'express_rate' => 'required|numeric|min:0',
            'priority_rate' => 'required|numeric|min:0',
            'free_shipping_threshold' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'integer'
        ]);

        ShippingZone::create($validated);

        return redirect()->route('admin.shipping-zones.index')
            ->with('success', 'Shipping zone created successfully');
    }

    /**
     * Show the form for editing the specified shipping zone
     */
    public function edit(ShippingZone $shippingZone)
    {
        $states = ShippingZone::getStates();
        return view('admin.shipping-zones.edit', compact('shippingZone', 'states'));
    }

    /**
     * Update the specified shipping zone
     */
    public function update(Request $request, ShippingZone $shippingZone)
    {
        $validated = $request->validate([
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:10',
            'postcode_range' => 'nullable|string|max:50',
            'standard_rate' => 'required|numeric|min:0',
            'express_rate' => 'required|numeric|min:0',
            'priority_rate' => 'required|numeric|min:0',
            'free_shipping_threshold' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
            'sort_order' => 'integer'
        ]);

        $shippingZone->update($validated);

        return redirect()->route('admin.shipping-zones.index')
            ->with('success', 'Shipping zone updated successfully');
    }

    /**
     * Remove the specified shipping zone
     */
    public function destroy(ShippingZone $shippingZone)
    {
        // Don't allow deletion of default zone
        if ($shippingZone->city === 'Other' && $shippingZone->state === 'ALL') {
            return redirect()->route('admin.shipping-zones.index')
                ->with('error', 'Cannot delete the default shipping zone');
        }

        $shippingZone->delete();

        return redirect()->route('admin.shipping-zones.index')
            ->with('success', 'Shipping zone deleted successfully');
    }

    /**
     * Toggle active status
     */
    public function toggleActive(ShippingZone $shippingZone)
    {
        $shippingZone->is_active = !$shippingZone->is_active;
        $shippingZone->save();

        return response()->json([
            'success' => true,
            'is_active' => $shippingZone->is_active
        ]);
    }

    /**
     * Bulk update shipping rates
     */
    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:percentage,fixed',
            'value' => 'required|numeric',
            'zones' => 'required|array',
            'zones.*' => 'exists:shipping_zones,id'
        ]);

        $zones = ShippingZone::whereIn('id', $validated['zones'])->get();

        foreach ($zones as $zone) {
            if ($validated['type'] === 'percentage') {
                $zone->standard_rate = $zone->standard_rate * (1 + $validated['value'] / 100);
                $zone->express_rate = $zone->express_rate * (1 + $validated['value'] / 100);
                $zone->priority_rate = $zone->priority_rate * (1 + $validated['value'] / 100);
            } else {
                $zone->standard_rate += $validated['value'];
                $zone->express_rate += $validated['value'];
                $zone->priority_rate += $validated['value'];
            }
            $zone->save();
        }

        return redirect()->route('admin.shipping-zones.index')
            ->with('success', 'Shipping rates updated for ' . count($zones) . ' zones');
    }
}