<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StyleGuide;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Helpers\ImageHelper;

class StyleGuideController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $styleGuides = StyleGuide::ordered()->paginate(10);
        return view('admin.style-guides.index', compact('styleGuides'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.style-guides.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:style_guides,slug',
            'tagline' => 'required|string|max:255',
            'description' => 'required|string',
            'region' => 'required|string|max:255',
            'population' => 'required|string|max:255',
            'fashion_score' => 'required|integer|min:1|max:10',
            'streetwear_popularity' => 'required|integer|min:1|max:100',
            'meta_title' => 'required|string',
            'meta_description' => 'required|string',
            'keywords' => 'required|string',
            'climate' => 'required|string|max:255',
            'vibe' => 'required|string|max:255',
            'hero_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'content' => 'nullable|string',
            'styling_tips' => 'nullable|array',
            'styling_tips.*' => 'string',
            'style_characteristics' => 'nullable|array',
            'style_characteristics.*.title' => 'required_with:style_characteristics|string',
            'style_characteristics.*.description' => 'required_with:style_characteristics|string',
            'key_pieces' => 'nullable|array',
            'key_pieces.*' => 'string',
            'seasonal_styling' => 'nullable|array',
            'cultural_influences' => 'nullable|array',
            'cultural_influences.*.aspect' => 'required_with:cultural_influences|string',
            'cultural_influences.*.description' => 'required_with:cultural_influences|string',
            'color_palette' => 'nullable|array',
            'color_palette.*.name' => 'required_with:color_palette|string',
            'color_palette.*.hex' => 'required_with:color_palette|string',
            'shopping_areas' => 'nullable|array',
            'shopping_areas.*.name' => 'required_with:shopping_areas|string',
            'shopping_areas.*.description' => 'required_with:shopping_areas|string',
            'weather' => 'nullable|array',
            'style_gallery' => 'nullable|array',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }

        // Handle hero image upload
        if ($request->hasFile('hero_image')) {
            $file = $request->file('hero_image');
            $filename = time() . '_' . Str::slug($data['name']) . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('style-guides', $filename, 'public');
            $data['hero_image'] = $path;
        }

        // Clean up JSON fields
        $data = $this->processJsonFields($data);

        StyleGuide::create($data);

        return redirect()
            ->route('admin.style-guides.index')
            ->with('success', 'Style guide created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(StyleGuide $styleGuide)
    {
        return view('admin.style-guides.show', compact('styleGuide'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(StyleGuide $styleGuide)
    {
        return view('admin.style-guides.edit', compact('styleGuide'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, StyleGuide $styleGuide)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:style_guides,slug,' . $styleGuide->id,
            'tagline' => 'required|string|max:255',
            'description' => 'required|string',
            'region' => 'required|string|max:255',
            'population' => 'required|string|max:255',
            'fashion_score' => 'required|integer|min:1|max:10',
            'streetwear_popularity' => 'required|integer|min:1|max:100',
            'meta_title' => 'required|string',
            'meta_description' => 'required|string',
            'keywords' => 'required|string',
            'climate' => 'required|string|max:255',
            'vibe' => 'required|string|max:255',
            'hero_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'content' => 'nullable|string',
            'styling_tips' => 'nullable|array',
            'styling_tips.*' => 'string',
            'style_characteristics' => 'nullable|array',
            'style_characteristics.*.title' => 'required_with:style_characteristics|string',
            'style_characteristics.*.description' => 'required_with:style_characteristics|string',
            'key_pieces' => 'nullable|array',
            'key_pieces.*' => 'string',
            'seasonal_styling' => 'nullable|array',
            'cultural_influences' => 'nullable|array',
            'cultural_influences.*.aspect' => 'required_with:cultural_influences|string',
            'cultural_influences.*.description' => 'required_with:cultural_influences|string',
            'color_palette' => 'nullable|array',
            'color_palette.*.name' => 'required_with:color_palette|string',
            'color_palette.*.hex' => 'required_with:color_palette|string',
            'shopping_areas' => 'nullable|array',
            'shopping_areas.*.name' => 'required_with:shopping_areas|string',
            'shopping_areas.*.description' => 'required_with:shopping_areas|string',
            'weather' => 'nullable|array',
            'style_gallery' => 'nullable|array',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }

        // Handle hero image upload
        if ($request->hasFile('hero_image')) {
            // Delete old image if exists
            if ($styleGuide->hero_image && Storage::disk('public')->exists($styleGuide->hero_image)) {
                Storage::disk('public')->delete($styleGuide->hero_image);
            }

            $file = $request->file('hero_image');
            $filename = time() . '_' . Str::slug($data['name']) . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('style-guides', $filename, 'public');
            $data['hero_image'] = $path;
        }

        // Clean up JSON fields
        $data = $this->processJsonFields($data);

        $styleGuide->update($data);

        return redirect()
            ->route('admin.style-guides.index')
            ->with('success', 'Style guide updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(StyleGuide $styleGuide)
    {
        // Delete associated images
        if ($styleGuide->hero_image && Storage::disk('public')->exists($styleGuide->hero_image)) {
            Storage::disk('public')->delete($styleGuide->hero_image);
        }

        // Delete style gallery images if they exist
        if ($styleGuide->style_gallery) {
            foreach ($styleGuide->style_gallery as $galleryItem) {
                if (isset($galleryItem['image']) && Storage::disk('public')->exists($galleryItem['image'])) {
                    Storage::disk('public')->delete($galleryItem['image']);
                }
            }
        }

        $styleGuide->delete();

        return redirect()
            ->route('admin.style-guides.index')
            ->with('success', 'Style guide deleted successfully!');
    }

    /**
     * Toggle active status of a style guide
     */
    public function toggleActive(StyleGuide $styleGuide)
    {
        $styleGuide->update([
            'is_active' => !$styleGuide->is_active
        ]);

        $status = $styleGuide->is_active ? 'activated' : 'deactivated';
        
        return response()->json([
            'success' => true,
            'message' => "Style guide {$status} successfully!",
            'is_active' => $styleGuide->is_active
        ]);
    }

    /**
     * Duplicate a style guide
     */
    public function duplicate(StyleGuide $styleGuide)
    {
        $newStyleGuide = $styleGuide->replicate();
        $newStyleGuide->name = $newStyleGuide->name . ' (Copy)';
        $newStyleGuide->slug = Str::slug($newStyleGuide->name);
        $newStyleGuide->is_active = false;
        $newStyleGuide->save();

        return redirect()
            ->route('admin.style-guides.edit', $newStyleGuide)
            ->with('success', 'Style guide duplicated successfully! Please review and update the details.');
    }

    /**
     * Process JSON fields to clean up empty values
     */
    private function processJsonFields($data)
    {
        $jsonFields = [
            'styling_tips', 'style_characteristics', 'key_pieces', 
            'seasonal_styling', 'cultural_influences', 'color_palette',
            'shopping_areas', 'weather', 'style_gallery'
        ];

        foreach ($jsonFields as $field) {
            if (isset($data[$field])) {
                // Remove empty values from arrays
                $data[$field] = array_filter($data[$field], function($value) {
                    if (is_array($value)) {
                        return !empty(array_filter($value));
                    }
                    return !empty(trim($value));
                });

                // Reset array keys
                $data[$field] = array_values($data[$field]);

                // Set to null if empty
                if (empty($data[$field])) {
                    $data[$field] = null;
                }
            }
        }

        return $data;
    }
}
