<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TaxRate;
use App\Models\ShippingZone;
use Illuminate\Http\Request;

class TaxRateController extends Controller
{
    /**
     * Display a listing of tax rates
     */
    public function index()
    {
        $taxRates = TaxRate::orderBy('state')
            ->orderBy('city')
            ->paginate(20);
        
        return view('admin.tax-rates.index', compact('taxRates'));
    }

    /**
     * Show the form for creating a new tax rate
     */
    public function create()
    {
        $states = ShippingZone::getStates();
        return view('admin.tax-rates.create', compact('states'));
    }

    /**
     * Store a newly created tax rate
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:10',
            'tax_name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0|max:100',
            'is_percentage' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer'
        ]);

        TaxRate::create($validated);

        return redirect()->route('admin.tax-rates.index')
            ->with('success', 'Tax rate created successfully');
    }

    /**
     * Show the form for editing the specified tax rate
     */
    public function edit(TaxRate $taxRate)
    {
        $states = ShippingZone::getStates();
        return view('admin.tax-rates.edit', compact('taxRate', 'states'));
    }

    /**
     * Update the specified tax rate
     */
    public function update(Request $request, TaxRate $taxRate)
    {
        $validated = $request->validate([
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:10',
            'tax_name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0|max:100',
            'is_percentage' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer'
        ]);

        $taxRate->update($validated);

        return redirect()->route('admin.tax-rates.index')
            ->with('success', 'Tax rate updated successfully');
    }

    /**
     * Remove the specified tax rate
     */
    public function destroy(TaxRate $taxRate)
    {
        // Don't allow deletion of default tax rate
        if ($taxRate->city === 'Other' && $taxRate->state === 'ALL') {
            return redirect()->route('admin.tax-rates.index')
                ->with('error', 'Cannot delete the default tax rate');
        }

        $taxRate->delete();

        return redirect()->route('admin.tax-rates.index')
            ->with('success', 'Tax rate deleted successfully');
    }

    /**
     * Toggle active status
     */
    public function toggleActive(TaxRate $taxRate)
    {
        $taxRate->is_active = !$taxRate->is_active;
        $taxRate->save();

        return response()->json([
            'success' => true,
            'is_active' => $taxRate->is_active
        ]);
    }
}