<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Cart;
use App\Models\Product;
use App\Models\ShippingAddress;
use App\Models\UserDiscount;
use App\Models\ShippingZone;
use App\Models\TaxRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Mail;
use App\Mail\OrderConfirmation;
use Stripe\Stripe;
use Stripe\PaymentIntent;
use Stripe\Exception\ApiErrorException;

class CheckoutController extends Controller
{
    public function __construct()
    {
        // Only set API key if configured
        if (config('services.stripe.secret')) {
            Stripe::setApiKey(config('services.stripe.secret'));
        }
    }

    /**
     * Show checkout page with dynamic shipping & tax calculations.
     */
    public function index()
    {
        // Get cart items for logged-in user OR current session (keeps both if user logged in with session cart merged)
        $cartItems = Cart::with('product')
            ->where(function ($q) {
                $q->where('user_id', Auth::id())
                  ->orWhere('session_id', session()->getId());
            })
            ->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')
                ->with('error', 'Your cart is empty');
        }

        // Subtotal
        $subtotal = $cartItems->sum(function ($item) {
            return $item->price * $item->quantity;
        });

        // Shipping info from session (defaults)
        $shippingCity = Session::get('shipping_city', 'Sydney');
        $shippingState = Session::get('shipping_state', 'NSW');
        $shippingMethod = Session::get('shipping_method', 'standard');

        // Tax (dynamic)
        $taxRate = null;
        $tax = 0.0;
        if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
            $taxRate = TaxRate::findByCity($shippingCity, $shippingState);
        }
        if ($taxRate && method_exists($taxRate, 'calculateTax')) {
            $tax = $taxRate->calculateTax($subtotal);
        } else {
            $tax = $subtotal * 0.10; // fallback 10%
        }

        // Shipping (dynamic)
        $shippingZone = null;
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
            $shippingZone = ShippingZone::findByCity($shippingCity, $shippingState);
        }
        if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
            $shippingZone = ShippingZone::getDefault();
        }
        if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
            $shipping = $shippingZone->getShippingRate($shippingMethod, $subtotal);
        } else {
            $shipping = $this->calculateShipping($subtotal, $shippingCity, $shippingState, $shippingMethod);
        }

        // Discount
        $discount = $this->calculateDiscountAmount($subtotal);
        $total = max(0, $subtotal + $tax + $shipping - $discount);

        // Saved addresses for logged in users
        $savedAddresses = [];
        if (Auth::check()) {
            $savedAddresses = ShippingAddress::where('user_id', Auth::id())->get();
        }

        // All shipping zones for selector (if supported)
        $shippingZones = [];
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'active')) {
            $shippingZones = ShippingZone::active()
                ->orderBy('state')
                ->orderBy('city')
                ->get();
        }

        return view('shop.checkout', compact(
            'cartItems', 'subtotal', 'tax', 'shipping', 'discount', 'total',
            'savedAddresses', 'shippingZones', 'shippingCity', 'shippingState',
            'shippingMethod', 'taxRate', 'shippingZone'
        ));
    }

    /**
     * Process checkout: validate, create order + items, update stocks, clear cart, and proceed to payment.
     */
    public function process(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address_line_1' => 'required|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'required|string|max:255',
            'state' => 'required|string|max:255',
            'postal_code' => 'required|string|max:20',
            'country' => 'required|string|max:255',
            'payment_method' => 'required|in:stripe,paypal,bank_transfer',
            'shipping_method' => 'required|in:standard,express,priority',
            'save_address' => 'nullable|boolean',
            'notes' => 'nullable|string'
        ]);

        DB::beginTransaction();

        try {
            // Retrieve cart items
            $cartItems = Cart::with('product')
                ->where(function ($q) {
                    $q->where('user_id', Auth::id())
                      ->orWhere('session_id', session()->getId());
                })
                ->get();

            if ($cartItems->isEmpty()) {
                throw new \Exception('Cart is empty');
            }

            // Subtotal
            $subtotal = $cartItems->sum(function ($item) {
                return $item->price * $item->quantity;
            });

            // Dynamic tax based on delivery city
            $taxRate = null;
            $tax = 0.0;
            if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
                $taxRate = TaxRate::findByCity($validated['city'], $validated['state']);
            }
            if ($taxRate && method_exists($taxRate, 'calculateTax')) {
                $tax = $taxRate->calculateTax($subtotal);
            } else {
                $tax = $subtotal * 0.10;
            }

            // Dynamic shipping based on delivery city
            $shippingZone = null;
            if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
                $shippingZone = ShippingZone::findByCity($validated['city'], $validated['state']);
            }
            if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
                $shippingZone = ShippingZone::getDefault();
            }
            if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
                $shipping = $shippingZone->getShippingRate($validated['shipping_method'], $subtotal);
            } else {
                $shipping = $this->calculateShipping($subtotal, $validated['city'], $validated['state'], $validated['shipping_method']);
            }

            // Discount
            $discount = $this->calculateDiscountAmount($subtotal);
            $total = max(0, $subtotal + $tax + $shipping - $discount);

            // Create order
            $order = Order::create([
                'order_number' => 'ORD-' . strtoupper(uniqid()),
                'user_id' => Auth::id(),
                'customer_name' => $validated['first_name'] . ' ' . $validated['last_name'],
                'customer_email' => $validated['email'],
                'customer_phone' => $validated['phone'],
                'subtotal' => $subtotal,
                'tax_amount' => $tax,
                'shipping_amount' => $shipping,
                'discount_amount' => $discount,
                'total_amount' => $total,
                'payment_method' => $validated['payment_method'],
                'payment_status' => 'pending',
                'status' => 'pending',
                'notes' => $validated['notes'] ?? null,
                'coupon_code' => session('coupon.code') ?? session('discount.code') ?? null,
                'shipping_method' => $validated['shipping_method'],
                'shipping_city' => $validated['city'],
                'shipping_state' => $validated['state'],
            ]);

            // Save shipping address - FIXED: Set both postcode and postal_code
            $shippingAddress = ShippingAddress::create([
                'user_id' => Auth::id(),
                'order_id' => $order->id,
                'first_name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'phone' => $validated['phone'],
                'address_line_1' => $validated['address_line_1'],
                'address_line_2' => $validated['address_line_2'],
                'city' => $validated['city'],
                'state' => $validated['state'],
                'postcode' => $validated['postal_code'],  // Set postcode (required field)
                'postal_code' => $validated['postal_code'],  // Also set postal_code for compatibility
                'country' => $validated['country'],
                'is_default' => $request->save_address ? true : false,
            ]);

            // Create order items and decrement stock
            foreach ($cartItems as $cartItem) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $cartItem->product_id,
                    'product_name' => optional($cartItem->product)->name ?? ($cartItem->product_name ?? null),
                    'product_sku' => optional($cartItem->product)->sku ?? null,
                    'product_image' => optional($cartItem->product)->featured_image ?? null,
                    'price' => $cartItem->price,
                    'quantity' => $cartItem->quantity,
                    // store both modern 'attributes' and legacy 'product_options' if available
                    'attributes' => $cartItem->attributes ?? null,
                    'product_options' => $cartItem->product_options ?? null,
                    'total' => $cartItem->price * $cartItem->quantity,
                ]);

                // Decrement product stock safely
                if ($cartItem->product) {
                    $cartItem->product->decrement('stock', $cartItem->quantity);
                }
            }

            // If user-specific discount was used, mark as used
            if (session('discount.code')) {
                $userDiscount = UserDiscount::where('discount_code', session('discount.code'))
                    ->where('email', $validated['email'])
                    ->first();

                if ($userDiscount && method_exists($userDiscount, 'markAsUsed')) {
                    $userDiscount->markAsUsed();
                }
            }

            DB::commit();

            // Clear cart and relevant session values
            Cart::where(function ($q) {
                $q->where('user_id', Auth::id())
                  ->orWhere('session_id', session()->getId());
            })->delete();

            session()->forget(['coupon', 'discount', 'shipping_city', 'shipping_state', 'shipping_method']);

            // Redirect to appropriate payment handling
            if ($validated['payment_method'] === 'stripe') {
                return $this->processStripePayment($order);
            } elseif ($validated['payment_method'] === 'paypal') {
                // If you have PayPal flow, handle here
                return redirect()->route('checkout.payment', $order->order_number);
            } else {
                // Bank transfer / offline
                return redirect()->route('checkout.success', $order->order_number)
                    ->with('info', 'Bank transfer instructions have been sent to your email.');
            }

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Something went wrong: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Create Stripe Payment Intent and return stripe payment view.
     */
    private function processStripePayment($order)
    {
        try {
            $paymentIntent = PaymentIntent::create([
                'amount' => round($order->total_amount * 100), // cents
                'currency' => config('cashier.currency', 'usd'), // use your configured currency or default
                'metadata' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number
                ],
                'description' => 'Order #' . $order->order_number,
            ]);

            // Save intent id to order
            $order->update(['stripe_payment_intent' => $paymentIntent->id]);

            return view('shop.stripe-payment', [
                'order' => $order,
                'clientSecret' => $paymentIntent->client_secret,
                'stripeKey' => config('services.stripe.key')
            ]);
        } catch (ApiErrorException $e) {
            return redirect()->back()
                ->with('error', 'Payment processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Confirm payment (called by client after payment completes or webhook).
     */
    public function confirmPayment(Request $request, $orderNumber)
    {
        $order = Order::where('order_number', $orderNumber)->firstOrFail();

        if ($request->payment_status === 'succeeded') {
            $order->update([
                'payment_status' => 'paid',
                'status' => 'processing',
                'transaction_id' => $request->payment_intent_id ?? $order->stripe_payment_intent,
                'paid_at' => now()
            ]);

            // Send confirmation email
            try {
                Mail::to($order->customer_email)->send(new OrderConfirmation($order));
            } catch (\Exception $e) {
                \Log::error('Failed to send order confirmation: ' . $e->getMessage());
            }

            return redirect()->route('checkout.success', $order->order_number)
                ->with('success', 'Payment successful! Your order has been confirmed.');
        }

        return redirect()->route('checkout.index')
            ->with('error', 'Payment failed. Please try again.');
    }

    /**
     * Success page after checkout.
     */
    public function success($orderNumber)
    {
        $order = Order::with(['items.product', 'shippingAddress'])
            ->where('order_number', $orderNumber)
            ->firstOrFail();

        // Security check for owner (guests allowed if session has order flag)
        if ($order->user_id && $order->user_id !== Auth::id()) {
            if (!session()->has('order_' . $orderNumber)) {
                abort(403);
            }
        }

        // Remember in session for guest users to view success page
        session(['order_' . $orderNumber => true]);

        return view('shop.order-success', compact('order'));
    }

    /**
     * Payment cancelled
     */
    public function cancel()
    {
        return redirect()->route('cart.index')
            ->with('info', 'Payment was cancelled. Your cart items are still saved.');
    }

    /**
     * Fallback shipping calculation (used if ShippingZone model/methods are not available).
     */
    private function calculateShipping($subtotal, $city = null, $state = null, $method = null)
    {
        // Example: free shipping over 100
        if ($subtotal >= 100) {
            return 0;
        }

        // Method-based adjustments
        if ($method === 'express') {
            return 20;
        } elseif ($method === 'priority') {
            return 30;
        }

        // Default flat rate
        return 10;
    }

    /**
     * Calculate discount amount based on session 'discount' and 'coupon' keys.
     * Mirrors behavior in CartController to keep consistent totals.
     */
    private function calculateDiscountAmount($subtotal)
    {
        $discountAmount = 0;

        // User-specific discount (session key: 'discount')
        $userDisc = Session::get('discount');
        if ($userDisc && is_array($userDisc)) {
            $type = $userDisc['type'] ?? 'percentage';
            if ($type === 'fixed') {
                $discountAmount = floatval($userDisc['percentage'] ?? ($userDisc['discount'] ?? 0));
            } else {
                $pct = floatval($userDisc['percentage'] ?? 0);
                $discountAmount = ($pct / 100) * $subtotal;
            }
            return round(max(0, $discountAmount), 2);
        }

        // Regular coupon (session key: 'coupon')
        $couponSess = Session::get('coupon');
        if ($couponSess && is_array($couponSess)) {
            $type = $couponSess['type'] ?? 'fixed';
            if ($type === 'fixed') {
                $discountAmount = floatval($couponSess['discount'] ?? 0);
            } else {
                $pct = floatval($couponSess['discount'] ?? 0);
                $discountAmount = ($pct / 100) * $subtotal;
            }
            return round(max(0, $discountAmount), 2);
        }

        // Backwards compatibility: coupon stored as string code
        if (Session::has('coupon') && !is_array(Session::get('coupon'))) {
            $code = Session::get('coupon');
            $coupon = \App\Models\Coupon::where('code', $code)->where('is_active', true)->first();
            if ($coupon && method_exists($coupon, 'isValid') && $coupon->isValid()) {
                if (method_exists($coupon, 'calculateDiscount')) {
                    return round($coupon->calculateDiscount($subtotal), 2);
                } else {
                    if (($coupon->discount_type ?? 'fixed') === 'percentage') {
                        return round(($coupon->discount_amount / 100) * $subtotal, 2);
                    }
                    return round($coupon->discount_amount, 2);
                }
            } else {
                Session::forget('coupon');
            }
        }

        return 0;
    }

    /**
     * Handle Stripe webhooks (payment success/failure).
     */
    public function handleWebhook(Request $request)
    {
        $endpoint_secret = config('services.stripe.webhook_secret');
        $payload = $request->getContent();
        $sig_header = $request->header('Stripe-Signature');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload, $sig_header, $endpoint_secret
            );
        } catch (\Exception $e) {
            return response('Webhook Error', 400);
        }

        switch ($event->type) {
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object;
                $order = Order::where('stripe_payment_intent', $paymentIntent->id)->first();

                if ($order && $order->payment_status !== 'paid') {
                    $order->update([
                        'payment_status' => 'paid',
                        'status' => 'processing',
                        'transaction_id' => $paymentIntent->id,
                        'paid_at' => now()
                    ]);

                    try {
                        Mail::to($order->customer_email)->send(new OrderConfirmation($order));
                    } catch (\Exception $e) {
                        \Log::error('Webhook email failed: ' . $e->getMessage());
                    }
                }
                break;

            case 'payment_intent.payment_failed':
                $paymentIntent = $event->data->object;
                $order = Order::where('stripe_payment_intent', $paymentIntent->id)->first();

                if ($order) {
                    $order->update([
                        'payment_status' => 'failed',
                        'status' => 'cancelled'
                    ]);
                }
                break;
        }

        return response('Webhook Handled', 200);
    }
}