<?php

namespace App\Http\Controllers;

use App\Models\NewsletterSubscriber;
use App\Models\UserDiscount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\NewsletterWelcome;

class NewsletterController extends Controller
{
    public function subscribe(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|unique:newsletter_subscribers,email',
            'name' => 'nullable|string|max:255'
        ]);

        // Create subscriber
        $subscriber = NewsletterSubscriber::create([
            'email' => $validated['email'],
            'name' => $validated['name'] ?? null,
            'subscribed_at' => now()
        ]);

        // Generate discount code
        $discountCode = $subscriber->generateDiscountCode();

        // Create user discount
        UserDiscount::create([
            'user_id' => auth()->id(),
            'email' => $validated['email'],
            'discount_code' => $discountCode,
            'discount_percentage' => 10,
            'discount_type' => 'percentage',
            'expires_at' => now()->addDays(30)
        ]);

        // Send welcome email with discount code
        try {
            Mail::to($validated['email'])->send(new NewsletterWelcome($subscriber, $discountCode));
        } catch (\Exception $e) {
            // Log error but don't fail the subscription
            \Log::error('Failed to send newsletter welcome email: ' . $e->getMessage());
        }

        // Set session for popup not to show again
        session(['newsletter_subscribed' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Successfully subscribed! Check your email for your 10% discount code.',
            'discount_code' => $discountCode
        ]);
    }

    public function checkSubscription(Request $request)
    {
        $email = $request->email ?? (auth()->check() ? auth()->user()->email : null);
        
        if (!$email) {
            return response()->json(['subscribed' => false]);
        }

        $subscriber = NewsletterSubscriber::where('email', $email)->first();
        
        return response()->json([
            'subscribed' => $subscriber ? true : false,
            'discount_code' => $subscriber ? $subscriber->discount_code : null
        ]);
    }
}