<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class CoreWebVitalsOptimizer
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);
        
        if ($response->headers->get('content-type') && 
            strpos($response->headers->get('content-type'), 'text/html') !== false) {
            
            $content = $response->getContent();
            
            // Add critical performance headers
            $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
            $response->headers->set('X-Content-Type-Options', 'nosniff');
            $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
            
            // Add resource hints for better loading
            $resourceHints = $this->generateResourceHints();
            $content = str_replace('<head>', '<head>' . $resourceHints, $content);
            
            // Optimize critical CSS loading
            $content = $this->optimizeCSSLoading($content);
            
            // Add performance monitoring script
            $content = $this->addPerformanceMonitoring($content);
            
            $response->setContent($content);
        }
        
        return $response;
    }
    
    /**
     * Generate resource hints for better loading performance
     */
    private function generateResourceHints()
    {
        return '
        <!-- DNS prefetch for external resources -->
        <link rel="dns-prefetch" href="//fonts.googleapis.com">
        <link rel="dns-prefetch" href="//fonts.gstatic.com">
        <link rel="dns-prefetch" href="//cdnjs.cloudflare.com">
        <link rel="dns-prefetch" href="//cdn.jsdelivr.net">
        
        <!-- Preconnect for critical external resources -->
        <link rel="preconnect" href="//fonts.googleapis.com" crossorigin>
        <link rel="preconnect" href="//fonts.gstatic.com" crossorigin>
        
        <!-- Preload critical resources -->
        <link rel="preload" href="/css/app.css" as="style">
        <link rel="preload" href="/js/app.js" as="script">
        ';
    }
    
    /**
     * Optimize CSS loading for better CLS
     */
    private function optimizeCSSLoading($content)
    {
        // Add critical CSS inline for above-the-fold content
        $criticalCSS = '
        <style>
            /* Critical CSS for Layout Shift prevention */
            .skeleton { background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%); background-size: 200% 100%; animation: loading 1.5s infinite; }
            @keyframes loading { 0% { background-position: 200% 0; } 100% { background-position: -200% 0; } }
            .img-placeholder { background-color: #f8f9fa; min-height: 200px; display: flex; align-items: center; justify-content: center; }
            .lazy-load { transition: opacity 0.3s ease-in-out; }
        </style>';
        
        $content = str_replace('</head>', $criticalCSS . '</head>', $content);
        
        return $content;
    }
    
    /**
     * Add Core Web Vitals monitoring
     */
    private function addPerformanceMonitoring($content)
    {
        $monitoringScript = '
        <script>
            // Core Web Vitals monitoring
            function sendToAnalytics(metric) {
                // Send to your analytics platform
                console.log("Core Web Vitals:", metric);
                
                // Example: Send to Google Analytics
                if (typeof gtag !== "undefined") {
                    gtag("event", metric.name, {
                        event_category: "Web Vitals",
                        value: Math.round(metric.name === "CLS" ? metric.value * 1000 : metric.value),
                        event_label: metric.id,
                        non_interaction: true,
                    });
                }
            }

            // Load web-vitals library and measure metrics
            const script = document.createElement("script");
            script.src = "https://unpkg.com/web-vitals@3/dist/web-vitals.iife.js";
            script.onload = function() {
                webVitals.getCLS(sendToAnalytics);
                webVitals.getFID(sendToAnalytics);
                webVitals.getFCP(sendToAnalytics);
                webVitals.getLCP(sendToAnalytics);
                webVitals.getTTFB(sendToAnalytics);
            };
            document.head.appendChild(script);
        </script>';
        
        $content = str_replace('</body>', $monitoringScript . '</body>', $content);
        
        return $content;
    }
}