<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CommunityPost extends Model
{
    use HasFactory;
    
    protected $fillable = [
        'name',
        'location',
        'caption',
        'products_featured',
        'instagram_handle',
        'image_path',
        'consent_gallery',
        'consent_social',
        'consent_marketing',
        'status',
        'featured',
        'likes',
        'approved_at'
    ];
    
    protected $casts = [
        'consent_gallery' => 'boolean',
        'consent_social' => 'boolean',
        'consent_marketing' => 'boolean',
        'featured' => 'boolean',
        'approved_at' => 'datetime'
    ];
    
    // Scopes
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }
    
    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }
    
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }
    
    // Accessors
    public function getImageUrlAttribute()
    {
        if (class_exists('\App\Helpers\ImageHelper')) {
            return \App\Helpers\ImageHelper::getImageUrl($this->image_path);
        }
        
        return asset('images/' . $this->image_path);
    }
    
    public function getInstagramUrlAttribute()
    {
        if ($this->instagram_handle) {
            $handle = ltrim($this->instagram_handle, '@');
            return "https://instagram.com/{$handle}";
        }
        
        return null;
    }
    
    // Mutators
    public function setInstagramHandleAttribute($value)
    {
        $this->attributes['instagram_handle'] = ltrim($value, '@');
    }
    
    // Helper methods
    public function approve()
    {
        $this->update([
            'status' => 'approved',
            'approved_at' => now()
        ]);
    }
    
    public function reject()
    {
        $this->update([
            'status' => 'rejected'
        ]);
    }
    
    public function toggleFeatured()
    {
        $this->update([
            'featured' => !$this->featured
        ]);
    }
    
    public function incrementLikes()
    {
        $this->increment('likes');
    }
}
