<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class DiscountCampaign extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'discount_percentage',
        'starts_at',
        'ends_at',
        'is_active',
        'is_seasonal',
        'show_timer',
        'product_ids',
        'category_ids',
        'banner_image'
    ];

    protected $casts = [
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
        'is_active' => 'boolean',
        'is_seasonal' => 'boolean',
        'show_timer' => 'boolean',
        'product_ids' => 'array',
        'category_ids' => 'array',
        'discount_percentage' => 'decimal:2'
    ];

    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where('starts_at', '<=', now())
                    ->where('ends_at', '>', now());
    }

    public function scopeUpcoming($query)
    {
        return $query->where('is_active', true)
                    ->where('starts_at', '>', now());
    }

    public function isActive()
    {
        return $this->is_active && 
               $this->starts_at <= now() && 
               $this->ends_at > now();
    }

    public function getDiscountedProducts()
    {
        $query = Product::active();

        if ($this->product_ids && count($this->product_ids) > 0) {
            $query->whereIn('id', $this->product_ids);
        } elseif ($this->category_ids && count($this->category_ids) > 0) {
            $query->whereIn('category_id', $this->category_ids);
        }

        return $query->get();
    }

    public function getRemainingTime()
    {
        if (!$this->isActive()) {
            return null;
        }

        $now = Carbon::now();
        $end = Carbon::parse($this->ends_at);
        
        return [
            'days' => $now->diffInDays($end),
            'hours' => $now->copy()->addDays($now->diffInDays($end))->diffInHours($end),
            'minutes' => $now->copy()->addDays($now->diffInDays($end))->addHours($now->copy()->addDays($now->diffInDays($end))->diffInHours($end))->diffInMinutes($end),
            'seconds' => $now->diffInSeconds($end) % 60
        ];
    }

    /**
     * Products relationship through pivot table
     */
    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_discount_campaigns')
                    ->withPivot('original_price', 'sale_price')
                    ->withTimestamps();
    }

    /**
     * Check if campaign should show timer
     */
    public function shouldShowTimer()
    {
        return $this->is_seasonal && $this->show_timer && $this->isActive();
    }

    /**
     * Scope for campaigns that should show timer
     */
    public function scopeWithTimer($query)
    {
        return $query->where('is_seasonal', true)
                    ->where('show_timer', true)
                    ->where('is_active', true)
                    ->where('starts_at', '<=', now())
                    ->where('ends_at', '>', now());
    }
}