<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Product extends Model
{
    use HasFactory;

    protected $fillable = [
        'category_id',
        'name',
        'slug',
        'description',
        'short_description',
        'sku',
        'price',
        'compare_price',
        'cost_price',
        'stock',
        'featured_image',
        'hover_image',
        'gallery_images',
        'sizes',
        'colors',
        'brand',
        'weight',
        'dimensions',
        'is_featured',
        'is_active',
        'views',
        'sold_count',
        'meta_title',
        'meta_description',
        'meta_keywords'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'compare_price' => 'decimal:2',
        'cost_price' => 'decimal:2',
        'weight' => 'decimal:2',
        'stock' => 'integer',
        'views' => 'integer',
        'sold_count' => 'integer',
        'is_featured' => 'boolean',
        'is_active' => 'boolean',
        'gallery_images' => 'array',
        'sizes' => 'array',
        'colors' => 'array'
    ];

    /**
     * Get the category that owns the product.
     */
    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Get the order items for the product.
     */
    public function orderItems()
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the wishlists for the product.
     */
    public function wishlists()
    {
        return $this->hasMany(Wishlist::class);
    }

    /**
     * Get the carts for the product.
     */
    public function carts()
    {
        return $this->hasMany(Cart::class);
    }

    /**
     * Get the reviews for the product.
     */
    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    /**
     * Get approved reviews for the product.
     */
    public function approvedReviews()
    {
        return $this->hasMany(Review::class)->where('is_approved', true);
    }

    /**
     * Scope a query to only include active products.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include featured products.
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope a query to only include products in stock.
     */
    public function scopeInStock($query)
    {
        return $query->where('stock', '>', 0);
    }

    /**
     * Increment the view count.
     */
    public function incrementViews()
    {
        $this->increment('views');
    }
    
    /**
     * Get the discount percentage attribute.
     */
    public function getDiscountPercentageAttribute()
    {
        if ($this->compare_price && $this->compare_price > $this->price) {
            return round((($this->compare_price - $this->price) / $this->compare_price) * 100);
        }
        return 0;
    }
    
    /**
     * Check if product is on sale.
     */
    public function getIsOnSaleAttribute()
    {
        return $this->compare_price && $this->compare_price > $this->price;
    }

    /**
     * Get average rating.
     */
    public function getAverageRatingAttribute()
    {
        return $this->approvedReviews()->avg('rating') ?? 0;
    }

    /**
     * Get total reviews count.
     */
    public function getTotalReviewsAttribute()
    {
        return $this->approvedReviews()->count();
    }

    /**
     * Discount campaigns relationship through pivot table
     */
    public function discountCampaigns()
    {
        return $this->belongsToMany(DiscountCampaign::class, 'product_discount_campaigns')
                    ->withPivot('original_price', 'sale_price')
                    ->withTimestamps();
    }

    /**
     * Get active discount campaigns for this product
     */
    public function activeDiscountCampaigns()
    {
        return $this->discountCampaigns()
                    ->where('is_active', true)
                    ->where('starts_at', '<=', now())
                    ->where('ends_at', '>', now());
    }

    /**
     * Check if product is currently on sale through discount campaigns
     */
    public function getIsOnCampaignSaleAttribute()
    {
        return $this->activeDiscountCampaigns()->exists();
    }

    /**
     * Get the current sale price from active discount campaigns
     */
    public function getCurrentSalePriceAttribute()
    {
        $activeCampaign = $this->activeDiscountCampaigns()->first();
        if ($activeCampaign) {
            return $activeCampaign->pivot->sale_price;
        }
        return null;
    }

    /**
     * Get the best sale price (lowest) from all active campaigns
     */
    public function getBestSalePriceAttribute()
    {
        $activeCampaigns = $this->activeDiscountCampaigns()->get();
        if ($activeCampaigns->count() > 0) {
            return $activeCampaigns->min('pivot.sale_price');
        }
        return null;
    }

    /**
     * Get the final display price (sale price if available, otherwise regular price)
     */
    public function getFinalPriceAttribute()
    {
        return $this->best_sale_price ?? $this->price;
    }

    /**
     * Get discount percentage from campaign sales
     */
    public function getCampaignDiscountPercentageAttribute()
    {
        $bestSalePrice = $this->best_sale_price;
        if ($bestSalePrice && $this->price) {
            return round((($this->price - $bestSalePrice) / $this->price) * 100);
        }
        return 0;
    }

    /**
     * Scope to get products on campaign sale
     */
    public function scopeOnCampaignSale($query)
    {
        return $query->whereHas('activeDiscountCampaigns');
    }
}