<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ShippingZone extends Model
{
    use HasFactory;

    protected $fillable = [
        'city',
        'state',
        'postcode_range',
        'standard_rate',
        'express_rate',
        'priority_rate',
        'free_shipping_threshold',
        'is_active',
        'sort_order'
    ];

    protected $casts = [
        'standard_rate' => 'decimal:2',
        'express_rate' => 'decimal:2',
        'priority_rate' => 'decimal:2',
        'free_shipping_threshold' => 'decimal:2',
        'is_active' => 'boolean',
        'sort_order' => 'integer'
    ];

    /**
     * Get active shipping zones
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get shipping zone by city
     */
    public static function findByCity($city, $state = null)
    {
        $query = self::active()->where('city', $city);
        
        if ($state) {
            $query->where('state', $state);
        }
        
        return $query->first();
    }

    /**
     * Get shipping zone by postcode
     */
    public static function findByPostcode($postcode)
    {
        $zones = self::active()->whereNotNull('postcode_range')->get();
        
        foreach ($zones as $zone) {
            if ($zone->postcodeInRange($postcode)) {
                return $zone;
            }
        }
        
        return null;
    }

    /**
     * Check if a postcode is in the zone's range
     */
    public function postcodeInRange($postcode)
    {
        if (!$this->postcode_range) {
            return false;
        }
        
        if (strpos($this->postcode_range, '-') !== false) {
            list($min, $max) = explode('-', $this->postcode_range);
            return $postcode >= $min && $postcode <= $max;
        }
        
        return $postcode == $this->postcode_range;
    }

    /**
     * Get shipping rate based on method and subtotal
     */
    public function getShippingRate($method = 'standard', $subtotal = 0)
    {
        // Check if free shipping applies
        if ($this->free_shipping_threshold && $subtotal >= $this->free_shipping_threshold) {
            return 0;
        }
        
        switch ($method) {
            case 'express':
                return $this->express_rate;
            case 'priority':
                return $this->priority_rate;
            default:
                return $this->standard_rate;
        }
    }

    /**
     * Get default shipping zone
     */
    public static function getDefault()
    {
        return self::active()
            ->where('city', 'Other')
            ->where('state', 'ALL')
            ->first();
    }

    /**
     * Get all Australian states
     */
    public static function getStates()
    {
        return [
            'NSW' => 'New South Wales',
            'VIC' => 'Victoria',
            'QLD' => 'Queensland',
            'WA' => 'Western Australia',
            'SA' => 'South Australia',
            'TAS' => 'Tasmania',
            'NT' => 'Northern Territory',
            'ACT' => 'Australian Capital Territory'
        ];
    }
}