<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class TaxRate extends Model
{
    use HasFactory;

    protected $fillable = [
        'city',
        'state',
        'tax_name',
        'rate',
        'is_percentage',
        'is_active',
        'sort_order'
    ];

    protected $casts = [
        'rate' => 'decimal:2',
        'is_percentage' => 'boolean',
        'is_active' => 'boolean',
        'sort_order' => 'integer'
    ];

    /**
     * Get active tax rates
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get tax rate by city
     */
    public static function findByCity($city, $state = null)
    {
        $query = self::active()->where('city', $city);
        
        if ($state) {
            $query->where('state', $state);
        }
        
        $rate = $query->first();
        
        // Fall back to default if not found
        if (!$rate) {
            $rate = self::getDefault();
        }
        
        return $rate;
    }

    /**
     * Calculate tax amount
     */
    public function calculateTax($amount)
    {
        if ($this->is_percentage) {
            return $amount * ($this->rate / 100);
        }
        
        return $this->rate;
    }

    /**
     * Get default tax rate
     */
    public static function getDefault()
    {
        return self::active()
            ->where('city', 'Other')
            ->where('state', 'ALL')
            ->first();
    }

    /**
     * Get formatted rate
     */
    public function getFormattedRateAttribute()
    {
        if ($this->is_percentage) {
            return $this->rate . '%';
        }
        
        return '$' . number_format($this->rate, 2);
    }
}