<?php

namespace App\Services;

use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class ImageOptimizer
{
    /**
     * Optimize image for web delivery
     */
    public static function optimizeForWeb($imagePath, $quality = 85)
    {
        try {
            // Generate optimized version path
            $pathInfo = pathinfo($imagePath);
            $optimizedPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_optimized.' . $pathInfo['extension'];
            
            // Check if optimized version already exists
            if (Storage::exists('public/' . $optimizedPath)) {
                return $optimizedPath;
            }
            
            // Create optimized version
            $image = Image::make(Storage::path('public/' . $imagePath));
            
            // Compress based on file type
            if (in_array(strtolower($pathInfo['extension']), ['jpg', 'jpeg'])) {
                $image->encode('jpg', $quality);
            } elseif (strtolower($pathInfo['extension']) === 'png') {
                $image->encode('png');
            }
            
            // Resize if too large (max width 1200px for web)
            if ($image->width() > 1200) {
                $image->resize(1200, null, function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                });
            }
            
            // Save optimized version
            Storage::put('public/' . $optimizedPath, (string) $image->encode());
            
            return $optimizedPath;
            
        } catch (\Exception $e) {
            // Return original path if optimization fails
            return $imagePath;
        }
    }
    
    /**
     * Generate responsive image srcset
     */
    public static function generateResponsiveSizes($imagePath, $sizes = [400, 800, 1200])
    {
        $srcset = [];
        $pathInfo = pathinfo($imagePath);
        
        foreach ($sizes as $width) {
            try {
                $resizedPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '_' . $width . 'w.' . $pathInfo['extension'];
                
                if (!Storage::exists('public/' . $resizedPath)) {
                    $image = Image::make(Storage::path('public/' . $imagePath));
                    $image->resize($width, null, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    });
                    
                    Storage::put('public/' . $resizedPath, (string) $image->encode());
                }
                
                $srcset[] = asset('storage/' . $resizedPath) . ' ' . $width . 'w';
                
            } catch (\Exception $e) {
                continue;
            }
        }
        
        return implode(', ', $srcset);
    }
    
    /**
     * Generate WebP version of image
     */
    public static function generateWebP($imagePath)
    {
        try {
            $pathInfo = pathinfo($imagePath);
            $webpPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '.webp';
            
            if (Storage::exists('public/' . $webpPath)) {
                return $webpPath;
            }
            
            $image = Image::make(Storage::path('public/' . $imagePath));
            $image->encode('webp', 85);
            
            Storage::put('public/' . $webpPath, (string) $image);
            
            return $webpPath;
            
        } catch (\Exception $e) {
            return null;
        }
    }
}