<?php

namespace App\Services;

use App\Models\Product;
use App\Models\Category;
use App\Models\Blog;
use Illuminate\Support\Collection;

class InternalLinkingService
{
    /**
     * Get related products for internal linking
     */
    public static function getRelatedProducts($product, $limit = 4)
    {
        return Product::where('is_active', true)
            ->where('id', '!=', $product->id)
            ->where('category_id', $product->category_id)
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }
    
    /**
     * Get related blogs for internal linking
     */
    public static function getRelatedBlogs($blog, $limit = 3)
    {
        return Blog::where('is_published', true)
            ->where('id', '!=', $blog->id)
            ->where(function($query) use ($blog) {
                // Match by tags if available
                if (!empty($blog->tags)) {
                    foreach ($blog->tags as $tag) {
                        $query->orWhereJsonContains('tags', $tag);
                    }
                }
            })
            ->orWhere('is_featured', true)
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }
    
    /**
     * Get contextual product recommendations
     */
    public static function getContextualProducts($context = 'homepage', $limit = 8)
    {
        switch ($context) {
            case 'homepage':
                return Product::where('is_active', true)
                    ->where('is_featured', true)
                    ->limit($limit)
                    ->get();
                    
            case 'category':
                return Product::where('is_active', true)
                    ->inRandomOrder()
                    ->limit($limit)
                    ->get();
                    
            case 'blog':
                return Product::where('is_active', true)
                    ->where('is_featured', true)
                    ->limit($limit)
                    ->get();
                    
            default:
                return Product::where('is_active', true)
                    ->inRandomOrder()
                    ->limit($limit)
                    ->get();
        }
    }
    
    /**
     * Get trending/popular content for internal linking
     */
    public static function getTrendingContent()
    {
        return [
            'products' => Product::where('is_active', true)
                ->where('is_featured', true)
                ->limit(4)
                ->get(),
                
            'blogs' => Blog::where('is_published', true)
                ->where('is_featured', true)
                ->limit(3)
                ->get(),
                
            'categories' => Category::where('is_active', true)
                ->withCount('products')
                ->orderBy('products_count', 'desc')
                ->limit(6)
                ->get()
        ];
    }
    
    /**
     * Auto-link content within text (for blog content)
     */
    public static function autoLinkContent($content)
    {
        // Get products and categories for auto-linking
        $products = Product::where('is_active', true)->get(['name', 'slug']);
        $categories = Category::where('is_active', true)->get(['name', 'slug']);
        
        // Auto-link product names
        foreach ($products as $product) {
            $pattern = '/\b(' . preg_quote($product->name, '/') . ')\b(?![^<]*>)/i';
            $replacement = '<a href="' . route('shop.product', $product->slug) . '" class="internal-link product-link">$1</a>';
            $content = preg_replace($pattern, $replacement, $content, 1); // Only link first occurrence
        }
        
        // Auto-link category names
        foreach ($categories as $category) {
            $pattern = '/\b(' . preg_quote($category->name, '/') . ')\b(?![^<]*>)/i';
            $replacement = '<a href="' . route('shop.category', $category->slug) . '" class="internal-link category-link">$1</a>';
            $content = preg_replace($pattern, $replacement, $content, 1); // Only link first occurrence
        }
        
        return $content;
    }
    
    /**
     * Generate contextual navigation links
     */
    public static function getContextualNavigation($currentPage)
    {
        $navigation = [
            'primary' => [
                'Shop All' => route('shop.index'),
                'Featured Products' => route('shop.index', ['featured' => 1]),
                'New Arrivals' => route('shop.index', ['sort' => 'newest']),
                'Blog' => route('blogs.index')
            ],
            'secondary' => []
        ];
        
        // Add category-specific navigation
        Category::where('is_active', true)
            ->limit(6)
            ->get()
            ->each(function($category) use (&$navigation) {
                $navigation['secondary'][$category->name] = route('shop.category', $category->slug);
            });
            
        return $navigation;
    }
}