<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        // Add missing columns if they don't exist
        Schema::table('shipping_addresses', function (Blueprint $table) {
            if (!Schema::hasColumn('shipping_addresses', 'user_id')) {
                // nullable so existing rows won't break
                $table->unsignedBigInteger('user_id')->nullable()->after('id');
                // we'll add the foreign key below (in a separate call) to avoid issues on some DB engines
            }

            if (!Schema::hasColumn('shipping_addresses', 'phone')) {
                $table->string('phone')->nullable()->after('last_name');
            }

            // controller expects 'postal_code' but your migration created 'postcode'
            // keep 'postcode' and add postal_code to be compatible with code
            if (!Schema::hasColumn('shipping_addresses', 'postal_code')) {
                $table->string('postal_code')->nullable()->after('state');
            }

            if (!Schema::hasColumn('shipping_addresses', 'is_default')) {
                $table->boolean('is_default')->default(false)->after('postal_code');
            }
        });

        // Add foreign key to users table (if user_id exists and users table exists)
        try {
            if (Schema::hasColumn('shipping_addresses', 'user_id') && Schema::hasTable('users')) {
                // Only add the FK if it doesn't already exist. MySQL convention handled by drop/create.
                DB::statement('ALTER TABLE `shipping_addresses` ADD CONSTRAINT `shipping_addresses_user_id_foreign` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE');
            }
        } catch (\Throwable $e) {
            // ignore if constraint already exists or DB does not allow; we still have the column
        }

        // Make order_id nullable so you can have addresses not tied to an order.
        // This uses raw SQL because changing column nullability often requires doctrine/dbal.
        try {
            $driver = DB::getDriverName();
            if ($driver === 'mysql') {
                // Only modify if order_id column exists
                if (Schema::hasColumn('shipping_addresses', 'order_id')) {
                    DB::statement('ALTER TABLE `shipping_addresses` MODIFY `order_id` BIGINT UNSIGNED NULL;');
                }
            } elseif ($driver === 'pgsql') {
                if (Schema::hasColumn('shipping_addresses', 'order_id')) {
                    DB::statement('ALTER TABLE shipping_addresses ALTER COLUMN order_id DROP NOT NULL;');
                }
            }
            // For other drivers, we attempt nothing (you can install doctrine/dbal and use change())
        } catch (\Throwable $e) {
            // If it fails (common when doctrine/dbal not installed) we silently continue.
            // The columns we added above still solve the immediate "unknown column user_id" error.
        }
    }

    public function down(): void
    {
        // Remove columns we added if present
        Schema::table('shipping_addresses', function (Blueprint $table) {
            if (Schema::hasColumn('shipping_addresses', 'is_default')) {
                $table->dropColumn('is_default');
            }
            if (Schema::hasColumn('shipping_addresses', 'postal_code')) {
                $table->dropColumn('postal_code');
            }
            if (Schema::hasColumn('shipping_addresses', 'phone')) {
                $table->dropColumn('phone');
            }

            if (Schema::hasColumn('shipping_addresses', 'user_id')) {
                // drop FK if exists (best-effort)
                try {
                    $table->dropForeign(['user_id']);
                } catch (\Throwable $e) {
                    // ignore if constraint name differs or doesn't exist
                }
                $table->dropColumn('user_id');
            }
        });

        // Try to revert order_id back to NOT NULL (best-effort; may require doctrine/dbal)
        try {
            $driver = DB::getDriverName();
            if ($driver === 'mysql') {
                if (Schema::hasColumn('shipping_addresses', 'order_id')) {
                    // Only do this if you know your application can handle it (there must be no NULLs)
                    DB::statement('ALTER TABLE `shipping_addresses` MODIFY `order_id` BIGINT UNSIGNED NOT NULL;');
                }
            } elseif ($driver === 'pgsql') {
                if (Schema::hasColumn('shipping_addresses', 'order_id')) {
                    DB::statement('ALTER TABLE shipping_addresses ALTER COLUMN order_id SET NOT NULL;');
                }
            }
        } catch (\Throwable $e) {
            // ignore; manual cleanup might be required
        }
    }
};