@extends('layouts.app')

@section('title', 'Order Details - #' . $order->order_number)
@section('meta_description', 'View detailed information about your order')

@section('content')
<!-- Premium Account Hero -->
<section class="account-hero-premium">
    <div class="hero-overlay"></div>
    <div class="container">
        <div class="hero-content-account" data-aos="fade-up">
            <h1 class="hero-title-account">ORDER #{{ $order->order_number }}</h1>
            <p class="hero-subtitle-account">{{ $order->created_at->format('F d, Y') }}</p>
        </div>
    </div>
</section>

<!-- Order Details Content -->
<section class="account-section-premium">
    <div class="container">
        <div class="row">
            <!-- Premium Sidebar -->
            <div class="col-lg-3" data-aos="fade-right">
                @include('account.sidebar')
            </div>

            <!-- Order Details -->
            <div class="col-lg-9">
                <!-- Action Bar -->
                <div class="detail-action-bar" data-aos="fade-up">
                    <a href="{{ route('account.orders') }}" class="btn-back">
                        <i class="fas fa-arrow-left"></i>
                        Back to Orders
                    </a>
                    <div class="detail-actions">
                        @if($order->tracking_number)
                        <a href="{{ route('order.track.index') }}?tracking_number={{ $order->tracking_number }}" 
                           class="btn-detail-action">
                            <i class="fas fa-truck"></i>
                            Track Package
                        </a>
                        @endif
                        <a href="{{ route('order.invoice', $order->order_number) }}" 
                           class="btn-detail-action primary" target="_blank">
                            <i class="fas fa-download"></i>
                            Download Invoice
                        </a>
                    </div>
                </div>

                <!-- Order Status Timeline -->
                <div class="status-timeline-card" data-aos="fade-up" data-aos-delay="100">
                    <h3 class="card-title-premium">
                        <span class="title-icon"><i class="fas fa-chart-line"></i></span>
                        Order Status
                    </h3>
                    
                    <!-- Progress Bar -->
                    <div class="order-progress-premium">
                        @php
                            $statusProgress = [
                                'pending' => 20,
                                'processing' => 40,
                                'shipped' => 60,
                                'out_for_delivery' => 80,
                                'delivered' => 100
                            ];
                            $progress = $statusProgress[$order->status] ?? 0;
                        @endphp
                        <div class="progress-bar-wrapper">
                            <div class="progress-bar-fill" style="width: {{ $progress }}%">
                                <span class="progress-indicator"></span>
                            </div>
                        </div>
                        
                        <div class="progress-steps">
                            <div class="step {{ $order->status != 'cancelled' ? 'completed' : '' }}">
                                <div class="step-icon">
                                    <i class="fas fa-shopping-cart"></i>
                                </div>
                                <span class="step-label">Order Placed</span>
                            </div>
                            <div class="step {{ in_array($order->status, ['processing', 'shipped', 'out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                                <div class="step-icon">
                                    <i class="fas fa-box"></i>
                                </div>
                                <span class="step-label">Processing</span>
                            </div>
                            <div class="step {{ in_array($order->status, ['shipped', 'out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                                <div class="step-icon">
                                    <i class="fas fa-truck"></i>
                                </div>
                                <span class="step-label">Shipped</span>
                            </div>
                            <div class="step {{ in_array($order->status, ['out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                                <div class="step-icon">
                                    <i class="fas fa-shipping-fast"></i>
                                </div>
                                <span class="step-label">Out for Delivery</span>
                            </div>
                            <div class="step {{ $order->status == 'delivered' ? 'completed' : '' }}">
                                <div class="step-icon">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <span class="step-label">Delivered</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Status Info -->
                    <div class="status-info-grid">
                        <div class="status-info-item">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <span class="info-label">Current Status</span>
                                <span class="info-value">{{ ucfirst(str_replace('_', ' ', $order->status)) }}</span>
                            </div>
                        </div>
                        @if($order->tracking_number)
                        <div class="status-info-item">
                            <i class="fas fa-barcode"></i>
                            <div>
                                <span class="info-label">Tracking Number</span>
                                <span class="info-value">{{ $order->tracking_number }}</span>
                            </div>
                        </div>
                        @endif
                        @if($order->carrier)
                        <div class="status-info-item">
                            <i class="fas fa-truck-loading"></i>
                            <div>
                                <span class="info-label">Carrier</span>
                                <span class="info-value">{{ $order->carrier }}</span>
                            </div>
                        </div>
                        @endif
                        @if($order->estimated_delivery)
                        <div class="status-info-item">
                            <i class="fas fa-calendar-check"></i>
                            <div>
                                <span class="info-label">Est. Delivery</span>
                                <span class="info-value">{{ \Carbon\Carbon::parse($order->estimated_delivery)->format('F d, Y') }}</span>
                            </div>
                        </div>
                        @endif
                    </div>
                </div>

                <!-- Order Items -->
                <div class="items-card-premium" data-aos="fade-up" data-aos-delay="200">
                    <h3 class="card-title-premium">
                        <span class="title-icon"><i class="fas fa-shopping-bag"></i></span>
                        Order Items
                    </h3>
                    
                    <div class="items-list-premium">
                        @foreach($order->items as $item)
                        <div class="item-row-premium">
                            <div class="item-image">
                                @if($item->product_image)
                                <img src="{{ asset('storage/' . $item->product_image) }}" 
                                     alt="{{ $item->product_name }}">
                                @else
                                <div class="image-placeholder">
                                    <i class="fas fa-image"></i>
                                </div>
                                @endif
                            </div>
                            
                            <div class="item-details">
                                <h4 class="item-name">{{ $item->product_name }}</h4>
                                @if($item->product_options)
                                    @php $options = json_decode($item->product_options, true); @endphp
                                    @if($options)
                                    <div class="item-options">
                                        @if(isset($options['size']))
                                        <span class="option-badge">Size: {{ $options['size'] }}</span>
                                        @endif
                                        @if(isset($options['color']))
                                        <span class="option-badge">Color: {{ $options['color'] }}</span>
                                        @endif
                                    </div>
                                    @endif
                                @endif
                                <p class="item-sku">SKU: {{ $item->product_sku ?? 'N/A' }}</p>
                            </div>
                            
                            <div class="item-quantity">
                                <span class="qty-label">Quantity</span>
                                <span class="qty-value">{{ $item->quantity }}</span>
                            </div>
                            
                            <div class="item-price">
                                <span class="price-label">Price</span>
                                <span class="price-value">${{ number_format($item->price, 2) }}</span>
                            </div>
                            
                            <div class="item-total">
                                <span class="total-label">Total</span>
                                <span class="total-value">${{ number_format($item->subtotal, 2) }}</span>
                            </div>
                        </div>
                        @endforeach
                    </div>
                </div>

                <!-- Order Info Row -->
                <div class="row">
                    <!-- Shipping Address -->
                    <div class="col-lg-6" data-aos="fade-right" data-aos-delay="300">
                        <div class="info-card-premium">
                            <h3 class="card-title-premium">
                                <span class="title-icon"><i class="fas fa-map-marker-alt"></i></span>
                                Shipping Address
                            </h3>
                            @if($order->shippingAddress)
                            <div class="address-content">
                                <p class="address-name">
                                    {{ $order->shippingAddress->first_name }} {{ $order->shippingAddress->last_name }}
                                </p>
                                <p class="address-line">{{ $order->shippingAddress->address_line_1 }}</p>
                                @if($order->shippingAddress->address_line_2)
                                <p class="address-line">{{ $order->shippingAddress->address_line_2 }}</p>
                                @endif
                                <p class="address-line">
                                    {{ $order->shippingAddress->city }}, {{ $order->shippingAddress->state }} {{ $order->shippingAddress->postal_code }}
                                </p>
                                <p class="address-line">{{ $order->shippingAddress->country }}</p>
                                <p class="address-phone">
                                    <i class="fas fa-phone"></i>
                                    {{ $order->shippingAddress->phone }}
                                </p>
                            </div>
                            @endif
                        </div>
                    </div>
                    
                    <!-- Order Summary -->
                    <div class="col-lg-6" data-aos="fade-left" data-aos-delay="300">
                        <div class="info-card-premium">
                            <h3 class="card-title-premium">
                                <span class="title-icon"><i class="fas fa-receipt"></i></span>
                                Order Summary
                            </h3>
                            <div class="summary-content">
                                <div class="summary-row">
                                    <span class="summary-label">Subtotal</span>
                                    <span class="summary-value">${{ number_format($order->subtotal, 2) }}</span>
                                </div>
                                @if($order->discount_amount > 0)
                                <div class="summary-row discount">
                                    <span class="summary-label">Discount</span>
                                    <span class="summary-value">-${{ number_format($order->discount_amount, 2) }}</span>
                                </div>
                                @endif
                                <div class="summary-row">
                                    <span class="summary-label">
                                        Tax
                                        @if($taxRate && $taxRate->tax_name)
                                            ({{ $taxRate->tax_name }})
                                        @else
                                            ({{ $taxRate && $taxRate->is_percentage ? $taxRate->rate . '%' : 'Standard Rate' }})
                                        @endif
                                    </span>
                                    <span class="summary-value">${{ number_format($order->tax_amount, 2) }}</span>
                                </div>
                                <div class="summary-row">
                                    <span class="summary-label">
                                        Shipping
                                        @if($order->shipping_method)
                                            ({{ ucfirst($order->shipping_method) }} Shipping)
                                        @endif
                                    </span>
                                    <span class="summary-value">
                                        @if($order->shipping_amount == 0)
                                            <span class="free-shipping">FREE</span>
                                        @else
                                            ${{ number_format($order->shipping_amount, 2) }}
                                        @endif
                                    </span>
                                </div>
                                <div class="summary-divider"></div>
                                <div class="summary-row total">
                                    <span class="summary-label">Total</span>
                                    <span class="summary-value">${{ number_format($order->total_amount, 2) }}</span>
                                </div>
                                
                                <div class="payment-info">
                                    <div class="payment-method">
                                        <i class="fas fa-credit-card"></i>
                                        <span>{{ ucfirst($order->payment_method) }}</span>
                                    </div>
                                    <div class="payment-status-indicator {{ $order->payment_status }}">
                                        @if($order->payment_status == 'paid')
                                            <i class="fas fa-check-circle"></i>
                                        @else
                                            <i class="fas fa-clock"></i>
                                        @endif
                                        Payment {{ ucfirst($order->payment_status) }}
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

@push('styles')
<style>
/* Reuse hero styles from orders page */
.account-hero-premium {
    height: 30vh;
    min-height: 250px;
    background: linear-gradient(135deg, #000 0%, #333 100%);
    position: relative;
    display: flex;
    align-items: center;
    overflow: hidden;
}

.account-hero-premium::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: repeating-linear-gradient(
        45deg,
        transparent,
        transparent 35px,
        rgba(255,255,255,.02) 35px,
        rgba(255,255,255,.02) 70px
    );
}

.account-section-premium {
    padding: 4rem 0;
    background: #fafafa;
    min-height: 60vh;
}

/* Action Bar */
.detail-action-bar {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.btn-back {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    color: #666;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s;
}

.btn-back:hover {
    color: #000;
    transform: translateX(-5px);
}

.detail-actions {
    display: flex;
    gap: 0.75rem;
}

.btn-detail-action {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: white;
    color: #333;
    border: 1px solid #e0e0e0;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.875rem;
    transition: all 0.3s;
}

.btn-detail-action:hover {
    background: #f8f8f8;
    border-color: #000;
    color: #000;
    transform: translateY(-2px);
}

.btn-detail-action.primary {
    background: #000;
    color: white;
    border-color: #000;
}

.btn-detail-action.primary:hover {
    background: #333;
    border-color: #333;
}

/* Status Timeline Card */
.status-timeline-card,
.items-card-premium,
.info-card-premium {
    background: white;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
}

.card-title-premium {
    font-size: 1.25rem;
    font-weight: 700;
    color: #000;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.title-icon {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #000 0%, #333 100%);
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1rem;
}

/* Progress Bar */
.order-progress-premium {
    margin-bottom: 2rem;
}

.progress-bar-wrapper {
    height: 8px;
    background: #e0e0e0;
    border-radius: 10px;
    position: relative;
    margin-bottom: 3rem;
}

.progress-bar-fill {
    height: 100%;
    background: linear-gradient(90deg, #000 0%, #333 100%);
    border-radius: 10px;
    position: relative;
    transition: width 1s ease;
}

.progress-indicator {
    position: absolute;
    right: -10px;
    top: 50%;
    transform: translateY(-50%);
    width: 20px;
    height: 20px;
    background: #000;
    border: 3px solid white;
    border-radius: 50%;
    box-shadow: 0 2px 10px rgba(0,0,0,0.2);
}

.progress-steps {
    display: flex;
    justify-content: space-between;
}

.step {
    text-align: center;
    opacity: 0.4;
    transition: all 0.3s;
}

.step.completed {
    opacity: 1;
}

.step-icon {
    width: 50px;
    height: 50px;
    background: #f0f0f0;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 0.5rem;
    font-size: 1.25rem;
    color: #999;
    transition: all 0.3s;
}

.step.completed .step-icon {
    background: linear-gradient(135deg, #000 0%, #333 100%);
    color: white;
}

.step-label {
    font-size: 0.75rem;
    color: #666;
    font-weight: 600;
}

/* Status Info Grid */
.status-info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    padding: 1.5rem;
    background: #fafafa;
    border-radius: 10px;
}

.status-info-item {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.status-info-item i {
    font-size: 1.25rem;
    color: #666;
}

.info-label {
    display: block;
    font-size: 0.75rem;
    color: #999;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.25rem;
}

.info-value {
    display: block;
    font-weight: 600;
    color: #000;
}

/* Items List */
.items-list-premium {
    border: 1px solid #e0e0e0;
    border-radius: 10px;
    overflow: hidden;
}

.item-row-premium {
    display: grid;
    grid-template-columns: 80px 1fr auto auto auto;
    gap: 1.5rem;
    padding: 1.5rem;
    border-bottom: 1px solid #f0f0f0;
    align-items: center;
    transition: all 0.3s;
}

.item-row-premium:last-child {
    border-bottom: none;
}

.item-row-premium:hover {
    background: #fafafa;
}

.item-image img,
.image-placeholder {
    width: 80px;
    height: 80px;
    object-fit: cover;
    border: 1px solid #e0e0e0;
}

.image-placeholder {
    display: flex;
    align-items: center;
    justify-content: center;
    background: #f8f8f8;
    color: #ccc;
    font-size: 1.5rem;
}

.item-name {
    font-size: 1rem;
    font-weight: 600;
    color: #000;
    margin-bottom: 0.5rem;
}

.item-options {
    display: flex;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
}

.option-badge {
    padding: 0.25rem 0.5rem;
    background: #f0f0f0;
    font-size: 0.75rem;
    border-radius: 5px;
}

.item-sku {
    font-size: 0.875rem;
    color: #999;
    margin: 0;
}

.item-quantity,
.item-price,
.item-total {
    text-align: center;
}

.qty-label,
.price-label,
.total-label {
    display: block;
    font-size: 0.75rem;
    color: #999;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.25rem;
}

.qty-value,
.price-value,
.total-value {
    display: block;
    font-weight: 600;
    color: #000;
}

.total-value {
    font-size: 1.125rem;
}

/* Info Cards */
.address-content {
    color: #333;
    line-height: 1.8;
}

.address-name {
    font-weight: 600;
    color: #000;
    margin-bottom: 0.5rem;
}

.address-line {
    margin-bottom: 0.25rem;
}

.address-phone {
    margin-top: 1rem;
    padding-top: 1rem;
    border-top: 1px solid #f0f0f0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #666;
}

/* Summary Content */
.summary-row {
    display: flex;
    justify-content: space-between;
    padding: 0.75rem 0;
}

.summary-row.discount .summary-value {
    color: #28a745;
}

.summary-divider {
    height: 2px;
    background: #f0f0f0;
    margin: 1rem 0;
}

.summary-row.total {
    font-size: 1.25rem;
    font-weight: 700;
}

.free-shipping {
    color: #28a745;
    font-weight: 600;
}

.payment-info {
    margin-top: 1.5rem;
    padding-top: 1.5rem;
    border-top: 1px solid #f0f0f0;
}

.payment-method {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
    color: #666;
}

.payment-status-indicator {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-weight: 600;
    font-size: 0.875rem;
}

.payment-status-indicator.paid {
    background: #d4edda;
    color: #155724;
}

.payment-status-indicator.pending {
    background: #fff3cd;
    color: #856404;
}

/* Responsive */
@media (max-width: 991px) {
    .item-row-premium {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .item-image {
        display: none;
    }
}

@media (max-width: 576px) {
    .detail-action-bar {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .detail-actions {
        width: 100%;
    }
    
    .btn-detail-action {
        flex: 1;
        justify-content: center;
    }
    
    .progress-steps {
        overflow-x: auto;
        padding-bottom: 1rem;
    }
    
    .status-info-grid {
        grid-template-columns: 1fr;
    }
}
</style>
@endpush

@push('scripts')
<script>
// Animate progress bar on page load
document.addEventListener('DOMContentLoaded', function() {
    const progressBar = document.querySelector('.progress-bar-fill');
    if (progressBar) {
        setTimeout(() => {
            progressBar.style.transition = 'width 1.5s cubic-bezier(0.4, 0, 0.2, 1)';
        }, 100);
    }
    
    // Add hover effect to items
    document.querySelectorAll('.item-row-premium').forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.boxShadow = '0 5px 15px rgba(0,0,0,0.05)';
        });
        
        row.addEventListener('mouseleave', function() {
            this.style.boxShadow = 'none';
        });
    });
});
</script>
@endpush
@endsection