@extends('layouts.app')

@section('title', 'My Wishlist')
@section('meta_description', 'View and manage your favorite items')

@section('content')
<!-- Premium Account Hero -->
<section class="account-hero-premium">
    <div class="hero-overlay"></div>
    <div class="container">
        <div class="hero-content-account" data-aos="fade-up">
            <h1 class="hero-title-account">MY WISHLIST</h1>
            <p class="hero-subtitle-account">Your favorite items in one place</p>
        </div>
    </div>
</section>

<!-- Wishlist Content -->
<section class="account-section-premium">
    <div class="container">
        <div class="row">
            <!-- Premium Sidebar -->
            <div class="col-lg-3" data-aos="fade-right">
                @include('account.sidebar')
            </div>

            <!-- Wishlist Items -->
            <div class="col-lg-9">
                <!-- Wishlist Header -->
                <div class="wishlist-header-premium" data-aos="fade-up">
                    <h2 class="section-title-account">Saved Items</h2>
                    @if($wishlistItems->count() > 0)
                    <div class="wishlist-stats">
                        <span class="stat-item">
                            <i class="fas fa-heart"></i>
                            {{ $wishlistItems->total() }} Items
                        </span>
                        <span class="stat-item">
                            <i class="fas fa-tag"></i>
                            Total Value: ${{ number_format($wishlistItems->sum(function($item) { return $item->product ? $item->product->price : 0; }), 2) }}
                        </span>
                    </div>
                    @endif
                </div>

                @if($wishlistItems->count() > 0)
                <!-- Wishlist Grid -->
                <div class="wishlist-grid-premium">
                    @foreach($wishlistItems as $index => $item)
                    @if($item->product)
                    <div class="wishlist-card-premium" data-aos="zoom-in" data-aos-delay="{{ $index * 50 }}">
                        <!-- Remove Button -->
                        <form action="{{ route('account.wishlist.remove', $item->id) }}" method="POST" 
                              class="remove-wishlist-form">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="btn-remove-wishlist" title="Remove from wishlist">
                                <i class="fas fa-times"></i>
                            </button>
                        </form>
                        
                        <!-- Product Image -->
                        <div class="wishlist-image-wrapper">
                            @if($item->product->compare_price && $item->product->compare_price > $item->product->price)
                            <div class="discount-badge">
                                -{{ round((($item->product->compare_price - $item->product->price) / $item->product->compare_price) * 100) }}%
                            </div>
                            @endif
                            
                            @if($item->product->stock <= 0)
                            <div class="stock-badge out">
                                Out of Stock
                            </div>
                            @elseif($item->product->stock < 5)
                            <div class="stock-badge low">
                                Only {{ $item->product->stock }} Left
                            </div>
                            @endif
                            
                            <a href="{{ route('shop.product', $item->product->slug) }}" class="image-link">
                                <img src="{{ $item->product->featured_image ? asset('storage/' . $item->product->featured_image) : asset('images/placeholder-product.jpg') }}" 
                                     alt="{{ $item->product->name }}"
                                     class="wishlist-image">
                                <div class="image-overlay">
                                    <span class="view-product">
                                        <i class="fas fa-eye"></i>
                                        View Product
                                    </span>
                                </div>
                            </a>
                        </div>
                        
                        <!-- Product Details -->
                        <div class="wishlist-details">
                            <div class="product-category">
                                {{ $item->product->category->name ?? 'Uncategorized' }}
                            </div>
                            
                            <h3 class="product-name">
                                <a href="{{ route('shop.product', $item->product->slug) }}">
                                    {{ $item->product->name }}
                                </a>
                            </h3>
                            
                            <!-- Rating -->
                            <div class="product-rating">
                                @for($i = 1; $i <= 5; $i++)
                                <i class="fas fa-star {{ $i <= 4 ? 'active' : '' }}"></i>
                                @endfor
                                <span class="rating-count">(24)</span>
                            </div>
                            
                            <!-- Price -->
                            <div class="product-price-wrapper">
                                @if($item->product->compare_price && $item->product->compare_price > $item->product->price)
                                <span class="price-old">${{ number_format($item->product->compare_price, 2) }}</span>
                                @endif
                                <span class="price-current">${{ number_format($item->product->price, 2) }}</span>
                            </div>
                            
                            <!-- Actions -->
                            <div class="wishlist-actions">
                                @if($item->product->stock > 0)
                                <button class="btn-add-to-cart-premium" data-product-id="{{ $item->product->id }}">
                                    <i class="fas fa-shopping-bag"></i>
                                    Add to Cart
                                </button>
                                @else
                                <button class="btn-out-of-stock" disabled>
                                    <i class="fas fa-times-circle"></i>
                                    Out of Stock
                                </button>
                                @endif
                            </div>
                            
                            <!-- Added Date -->
                            <div class="added-date">
                                Added {{ $item->created_at->diffForHumans() }}
                            </div>
                        </div>
                    </div>
                    @else
                    <!-- Product Not Available -->
                    <div class="wishlist-card-premium unavailable" data-aos="zoom-in" data-aos-delay="{{ $index * 50 }}">
                        <div class="unavailable-content">
                            <i class="fas fa-exclamation-triangle"></i>
                            <p>Product no longer available</p>
                            <form action="{{ route('account.wishlist.remove', $item->id) }}" method="POST">
                                @csrf
                                @method('DELETE')
                                <button type="submit" class="btn-remove-unavailable">Remove</button>
                            </form>
                        </div>
                    </div>
                    @endif
                    @endforeach
                </div>
                
                <!-- Pagination -->
                <div class="pagination-wrapper" data-aos="fade-up">
                    {{ $wishlistItems->links('pagination::bootstrap-5') }}
                </div>
                
                <!-- Continue Shopping -->
                <div class="continue-shopping-section" data-aos="fade-up">
                    <a href="{{ route('shop.index') }}" class="btn-continue-shopping">
                        <i class="fas fa-arrow-left"></i>
                        Continue Shopping
                    </a>
                </div>
                @else
                <!-- Empty Wishlist -->
                <div class="empty-wishlist-premium" data-aos="zoom-in">
                    <div class="empty-icon">
                        <i class="fas fa-heart"></i>
                    </div>
                    <h3>Your Wishlist is Empty</h3>
                    <p>Save items you love to your wishlist and shop them anytime</p>
                    <a href="{{ route('shop.index') }}" class="btn-start-shopping-premium">
                        START SHOPPING
                        <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
                @endif
            </div>
        </div>
    </div>
</section>

@push('styles')
<style>
/* Hero Styles (reused) */
.account-hero-premium {
    height: 30vh;
    min-height: 250px;
    background: linear-gradient(135deg, #000 0%, #333 100%);
    position: relative;
    display: flex;
    align-items: center;
    overflow: hidden;
}

.account-hero-premium::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: repeating-linear-gradient(
        45deg,
        transparent,
        transparent 35px,
        rgba(255,255,255,.02) 35px,
        rgba(255,255,255,.02) 70px
    );
}

.hero-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: radial-gradient(circle at center, transparent 0%, rgba(0,0,0,0.4) 100%);
}

.hero-content-account {
    text-align: center;
    color: white;
    position: relative;
    z-index: 2;
}

.hero-title-account {
    font-size: clamp(2rem, 4vw, 3rem);
    font-weight: 900;
    letter-spacing: 0.1em;
    margin-bottom: 0.5rem;
    font-family: 'Space Grotesk', sans-serif;
}

.hero-subtitle-account {
    font-size: 1.125rem;
    opacity: 0.9;
}

.account-section-premium {
    padding: 4rem 0;
    background: #fafafa;
    min-height: 60vh;
}

/* Wishlist Header */
.wishlist-header-premium {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.section-title-account {
    font-size: 1.75rem;
    font-weight: 800;
    color: #000;
    margin: 0;
    font-family: 'Space Grotesk', sans-serif;
}

.wishlist-stats {
    display: flex;
    gap: 1.5rem;
}

.stat-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: white;
    border: 1px solid #e0e0e0;
    font-size: 0.875rem;
    font-weight: 600;
    color: #666;
}

.stat-item i {
    color: #ff3366;
}

/* Wishlist Grid */
.wishlist-grid-premium {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 2rem;
    margin-bottom: 3rem;
}

.wishlist-card-premium {
    background: white;
    position: relative;
    overflow: hidden;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
}

.wishlist-card-premium:hover {
    transform: translateY(-10px);
    box-shadow: 0 15px 40px rgba(0,0,0,0.1);
}

.wishlist-card-premium::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 3px;
    background: linear-gradient(90deg, #ff3366 0%, #ff0066 100%);
    transform: scaleX(0);
    transition: transform 0.3s;
}

.wishlist-card-premium:hover::before {
    transform: scaleX(1);
}

/* Remove Button */
.btn-remove-wishlist {
    position: absolute;
    top: 1rem;
    right: 1rem;
    width: 35px;
    height: 35px;
    background: white;
    border: 1px solid #e0e0e0;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    z-index: 2;
    transition: all 0.3s;
}

.btn-remove-wishlist:hover {
    background: #ff3366;
    border-color: #ff3366;
    color: white;
    transform: rotate(90deg);
}

/* Image Wrapper */
.wishlist-image-wrapper {
    position: relative;
    aspect-ratio: 3/4;
    overflow: hidden;
    background: #f5f5f5;
}

.wishlist-image {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.5s;
}

.wishlist-card-premium:hover .wishlist-image {
    transform: scale(1.1);
}

.image-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: opacity 0.3s;
}

.wishlist-card-premium:hover .image-overlay {
    opacity: 1;
}

.view-product {
    background: white;
    color: #000;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transform: translateY(20px);
    transition: transform 0.3s;
}

.wishlist-card-premium:hover .view-product {
    transform: translateY(0);
}

/* Badges */
.discount-badge {
    position: absolute;
    top: 1rem;
    left: 1rem;
    background: #ff3366;
    color: white;
    padding: 0.25rem 0.75rem;
    font-size: 0.75rem;
    font-weight: 600;
    z-index: 2;
}

.stock-badge {
    position: absolute;
    bottom: 1rem;
    left: 1rem;
    padding: 0.25rem 0.75rem;
    font-size: 0.75rem;
    font-weight: 600;
    z-index: 2;
}

.stock-badge.out {
    background: #f44336;
    color: white;
}

.stock-badge.low {
    background: #FFB800;
    color: white;
}

/* Product Details */
.wishlist-details {
    padding: 1.5rem;
}

.product-category {
    font-size: 0.75rem;
    color: #999;
    text-transform: uppercase;
    letter-spacing: 0.1em;
    margin-bottom: 0.5rem;
}

.product-name {
    font-size: 1rem;
    font-weight: 600;
    margin-bottom: 0.75rem;
}

.product-name a {
    color: #000;
    text-decoration: none;
    transition: color 0.3s;
}

.product-name a:hover {
    color: #ff3366;
}

.product-rating {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    margin-bottom: 1rem;
}

.product-rating i {
    font-size: 0.875rem;
    color: #e0e0e0;
}

.product-rating i.active {
    color: #FFB800;
}

.rating-count {
    font-size: 0.75rem;
    color: #999;
    margin-left: 0.5rem;
}

.product-price-wrapper {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1.5rem;
}

.price-old {
    text-decoration: line-through;
    color: #999;
}

.price-current {
    font-size: 1.25rem;
    font-weight: 700;
    color: #000;
}

/* Actions */
.wishlist-actions {
    margin-bottom: 1rem;
}

.btn-add-to-cart-premium {
    width: 100%;
    padding: 0.875rem;
    background: #000;
    color: white;
    border: none;
    font-weight: 600;
    letter-spacing: 0.05em;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-add-to-cart-premium:hover {
    background: #333;
    transform: translateY(-2px);
}

.btn-out-of-stock {
    width: 100%;
    padding: 0.875rem;
    background: #f0f0f0;
    color: #999;
    border: none;
    font-weight: 600;
    cursor: not-allowed;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.added-date {
    font-size: 0.75rem;
    color: #999;
    text-align: center;
    padding-top: 1rem;
    border-top: 1px solid #f0f0f0;
}

/* Unavailable Card */
.wishlist-card-premium.unavailable {
    background: #f8f8f8;
}

.unavailable-content {
    padding: 3rem 1.5rem;
    text-align: center;
}

.unavailable-content i {
    font-size: 2rem;
    color: #999;
    margin-bottom: 1rem;
}

.unavailable-content p {
    color: #666;
    margin-bottom: 1rem;
}

.btn-remove-unavailable {
    padding: 0.5rem 1.5rem;
    background: white;
    border: 1px solid #e0e0e0;
    color: #666;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
}

.btn-remove-unavailable:hover {
    background: #ff3366;
    border-color: #ff3366;
    color: white;
}

/* Empty Wishlist */
.empty-wishlist-premium {
    background: white;
    padding: 5rem 2rem;
    text-align: center;
    box-shadow: 0 10px 30px rgba(0,0,0,0.05);
}

.empty-icon {
    width: 100px;
    height: 100px;
    margin: 0 auto 2rem;
    background: linear-gradient(135deg, #ff3366 0%, #ff0066 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 3rem;
    color: white;
    animation: heartbeat 2s infinite;
}

@keyframes heartbeat {
    0%, 100% { transform: scale(1); }
    25% { transform: scale(1.1); }
    50% { transform: scale(1); }
    75% { transform: scale(1.1); }
}

.empty-wishlist-premium h3 {
    font-size: 1.75rem;
    font-weight: 700;
    color: #333;
    margin-bottom: 0.5rem;
}

.empty-wishlist-premium p {
    color: #666;
    margin-bottom: 2rem;
}

.btn-start-shopping-premium {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem 2.5rem;
    background: #000;
    color: white;
    text-decoration: none;
    font-weight: 700;
    letter-spacing: 0.05em;
    transition: all 0.3s;
}

.btn-start-shopping-premium:hover {
    background: #333;
    transform: translateY(-2px);
    color: white;
}

/* Continue Shopping */
.continue-shopping-section {
    text-align: center;
    margin-top: 3rem;
}

.btn-continue-shopping {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem 2rem;
    background: white;
    color: #000;
    border: 2px solid #000;
    text-decoration: none;
    font-weight: 600;
    letter-spacing: 0.05em;
    transition: all 0.3s;
}

.btn-continue-shopping:hover {
    background: #000;
    color: white;
    transform: translateY(-2px);
}

/* Pagination */
.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 2rem;
}

/* Responsive */
@media (max-width: 991px) {
    .wishlist-grid-premium {
        grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
    }
}

@media (max-width: 576px) {
    .wishlist-header-premium {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .wishlist-stats {
        width: 100%;
        justify-content: space-between;
    }
    
    .wishlist-grid-premium {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
    }
}
</style>
@endpush

@push('scripts')
<script>
$(document).ready(function() {
    // Add to cart from wishlist
    $('.btn-add-to-cart-premium').click(function() {
        var productId = $(this).data('product-id');
        var button = $(this);
        var originalHtml = button.html();
        
        // Loading state
        button.html('<i class="fas fa-spinner fa-spin"></i> Adding...');
        button.prop('disabled', true);
        
        $.ajax({
            url: '{{ route("cart.add") }}',
            method: 'POST',
            data: {
                product_id: productId,
                quantity: 1,
                _token: '{{ csrf_token() }}'
            },
            success: function(response) {
                if(response.success) {
                    // Success animation
                    button.html('<i class="fas fa-check"></i> Added!');
                    button.addClass('success');
                    
                    // Show notification if toastr available
                    if(typeof toastr !== 'undefined') {
                        toastr.success('Product added to cart!');
                    }
                    
                    // Reset button after delay
                    setTimeout(function() {
                        button.html(originalHtml);
                        button.removeClass('success');
                        button.prop('disabled', false);
                    }, 2000);
                }
            },
            error: function(xhr) {
                button.html(originalHtml);
                button.prop('disabled', false);
                
                if(typeof toastr !== 'undefined') {
                    toastr.error('Failed to add product to cart');
                }
            }
        });
    });
    
    // Remove confirmation
    $('.remove-wishlist-form').on('submit', function(e) {
        e.preventDefault();
        var form = $(this);
        var card = form.closest('.wishlist-card-premium');
        
        if(confirm('Remove this item from your wishlist?')) {
            // Fade out animation
            card.fadeOut(300, function() {
                form.off('submit').submit();
            });
        }
    });
    
    // Image hover effect
    $('.wishlist-card-premium').hover(
        function() {
            $(this).find('.wishlist-image').css('transform', 'scale(1.1)');
        },
        function() {
            $(this).find('.wishlist-image').css('transform', 'scale(1)');
        }
    );
});
</script>
@endpush
@endsection