@extends('admin.layouts.admin')

@section('title', 'Shipping Zones')

@section('content')
<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h3 class="mb-0">Shipping Zones</h3>
                    <a href="{{ route('admin.shipping-zones.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Add New Zone
                    </a>
                </div>
                
                @if(session('success'))
                    <div class="alert alert-success alert-dismissible fade show mx-3 mt-3" role="alert">
                        {{ session('success') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                @endif

                @if(session('error'))
                    <div class="alert alert-danger alert-dismissible fade show mx-3 mt-3" role="alert">
                        {{ session('error') }}
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                @endif

                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>City</th>
                                    <th>State</th>
                                    <th>Postcode Range</th>
                                    <th>Standard Rate</th>
                                    <th>Express Rate</th>
                                    <th>Priority Rate</th>
                                    <th>Free Shipping</th>
                                    <th>Status</th>
                                    <th width="150">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($zones as $zone)
                                <tr>
                                    <td>
                                        <strong>{{ $zone->city }}</strong>
                                    </td>
                                    <td>{{ $zone->state }}</td>
                                    <td>{{ $zone->postcode_range ?: '-' }}</td>
                                    <td>
                                        @if($zone->standard_rate == 0)
                                            <span class="badge bg-success">FREE</span>
                                        @else
                                            ${{ number_format($zone->standard_rate, 2) }}
                                        @endif
                                    </td>
                                    <td>${{ number_format($zone->express_rate, 2) }}</td>
                                    <td>${{ number_format($zone->priority_rate, 2) }}</td>
                                    <td>
                                        @if($zone->free_shipping_threshold)
                                            <span class="text-success">
                                                ${{ number_format($zone->free_shipping_threshold, 2) }}+
                                            </span>
                                        @else
                                            <span class="text-muted">No free shipping</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="form-check form-switch">
                                            <input class="form-check-input toggle-active" 
                                                   type="checkbox" 
                                                   data-id="{{ $zone->id }}"
                                                   {{ $zone->is_active ? 'checked' : '' }}>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('admin.shipping-zones.edit', $zone) }}" 
                                               class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            @if(!($zone->city === 'Other' && $zone->state === 'ALL'))
                                            <form action="{{ route('admin.shipping-zones.destroy', $zone) }}" 
                                                  method="POST" 
                                                  onsubmit="return confirm('Are you sure?')"
                                                  style="display: inline;">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-sm btn-outline-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="9" class="text-center py-4">
                                        <p class="mb-0">No shipping zones found</p>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    {{ $zones->links() }}
                </div>
            </div>

            <!-- Bulk Update Card -->
            <div class="card mt-4">
                <div class="card-header">
                    <h5 class="mb-0">Bulk Rate Update</h5>
                </div>
                <div class="card-body">
                    <form action="{{ route('admin.shipping-zones.bulk-update') }}" method="POST" id="bulkUpdateForm">
                        @csrf
                        <div class="row">
                            <div class="col-md-3">
                                <label class="form-label">Update Type</label>
                                <select name="type" class="form-select" required>
                                    <option value="percentage">Percentage Increase/Decrease</option>
                                    <option value="fixed">Fixed Amount</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Value</label>
                                <input type="number" name="value" class="form-control" step="0.01" required>
                                <small class="text-muted">Use negative values to decrease</small>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Select Zones</label>
                                <select name="zones[]" class="form-select" multiple required>
                                    @foreach($zones as $zone)
                                    <option value="{{ $zone->id }}">{{ $zone->city }}, {{ $zone->state }}</option>
                                    @endforeach
                                </select>
                                <small class="text-muted">Hold Ctrl/Cmd to select multiple</small>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <button type="submit" class="btn btn-primary w-100">Update Rates</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.querySelectorAll('.toggle-active').forEach(function(toggle) {
    toggle.addEventListener('change', function() {
        const zoneId = this.dataset.id;
        
        fetch(`/admin/shipping-zones/${zoneId}/toggle-active`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (!data.success) {
                this.checked = !this.checked;
                alert('Failed to update status');
            }
        });
    });
});
</script>
@endpush