@extends('admin.layouts.admin')

@section('title', 'Create Fashion News Article')

@push('styles')
<script src="https://cdn.tiny.cloud/1/no-api-key/tinymce/6/tinymce.min.js" referrerpolicy="origin"></script>
@endpush

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3">Create New Fashion News Article</h1>
            <p class="text-muted">Add a new SEO-optimized fashion news article</p>
        </div>
        <a href="{{ route('admin.trends.news') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back to News
        </a>
    </div>

    <form action="{{ route('admin.trends.store-news') }}" method="POST" enctype="multipart/form-data">
        @csrf
        
        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8">
                <!-- Basic Information -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">Article Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Title -->
                        <div class="mb-3">
                            <label for="title" class="form-label">Article Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title') }}" 
                                   placeholder="e.g., Australian Fashion Week 2024: Streetwear Revolution">
                            <div class="form-text">Use relevant keywords for better SEO</div>
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Excerpt -->
                        <div class="mb-3">
                            <label for="excerpt" class="form-label">Article Excerpt *</label>
                            <textarea class="form-control @error('excerpt') is-invalid @enderror" 
                                      id="excerpt" name="excerpt" rows="3" maxlength="500"
                                      placeholder="Brief summary of the article for previews and SEO...">{{ old('excerpt') }}</textarea>
                            <div class="form-text">
                                <span id="excerptCount">0</span>/500 characters. This appears in search results and article previews.
                            </div>
                            @error('excerpt')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Content -->
                        <div class="mb-3">
                            <label for="content" class="form-label">Article Content *</label>
                            <textarea class="form-control @error('content') is-invalid @enderror" 
                                      id="content" name="content" rows="15">{{ old('content') }}</textarea>
                            <div class="form-text">Write comprehensive, SEO-friendly content (minimum 500 words)</div>
                            @error('content')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Image Upload -->
                        <div class="mb-3">
                            <label for="image" class="form-label">Featured Image *</label>
                            <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <div class="form-text">
                                Upload a high-quality image (JPEG, PNG, JPG, GIF). Recommended size: 1200x800px
                            </div>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            
                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="previewImg" src="" class="img-thumbnail" style="max-width: 300px;">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- SEO Section -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0">SEO Optimization</h5>
                    </div>
                    <div class="card-body">
                        <!-- Meta Title -->
                        <div class="mb-3">
                            <label for="meta_title" class="form-label">Meta Title *</label>
                            <input type="text" class="form-control @error('meta_title') is-invalid @enderror" 
                                   id="meta_title" name="meta_title" value="{{ old('meta_title') }}" 
                                   maxlength="255"
                                   placeholder="e.g., Australian Fashion Week 2024 | Latest Streetwear Trends">
                            <div class="form-text">
                                <span id="metaTitleCount">0</span>/255 characters. This appears in search engine results.
                            </div>
                            @error('meta_title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Meta Description -->
                        <div class="mb-3">
                            <label for="meta_description" class="form-label">Meta Description *</label>
                            <textarea class="form-control @error('meta_description') is-invalid @enderror" 
                                      id="meta_description" name="meta_description" rows="3" maxlength="300"
                                      placeholder="Compelling description that appears in search results...">{{ old('meta_description') }}</textarea>
                            <div class="form-text">
                                <span id="metaDescCount">0</span>/300 characters. This appears in search engine results.
                            </div>
                            @error('meta_description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Keywords -->
                        <div class="mb-3">
                            <label for="keywords" class="form-label">SEO Keywords *</label>
                            <input type="text" class="form-control @error('keywords') is-invalid @enderror" 
                                   id="keywords" name="keywords" value="{{ old('keywords') }}" 
                                   placeholder="e.g., Australian Fashion Week, streetwear trends, Melbourne fashion, Sydney fashion">
                            <div class="form-text">
                                Separate keywords with commas. Focus on Australian fashion terms.
                            </div>
                            @error('keywords')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="col-lg-4">
                <!-- Article Settings -->
                <div class="card mb-4 sticky-top" style="top: 20px;">
                    <div class="card-header">
                        <h6 class="mb-0">Article Settings</h6>
                    </div>
                    <div class="card-body">
                        <!-- Category -->
                        <div class="mb-3">
                            <label for="category" class="form-label">Category *</label>
                            <select class="form-select @error('category') is-invalid @enderror" 
                                    id="category" name="category">
                                <option value="">Select Category</option>
                                <option value="Brand News" {{ old('category') == 'Brand News' ? 'selected' : '' }}>
                                    🏢 Brand News
                                </option>
                                <option value="Industry News" {{ old('category') == 'Industry News' ? 'selected' : '' }}>
                                    🏭 Industry News
                                </option>
                                <option value="Global Trends" {{ old('category') == 'Global Trends' ? 'selected' : '' }}>
                                    🌍 Global Trends
                                </option>
                            </select>
                            @error('category')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Author -->
                        <div class="mb-3">
                            <label for="author" class="form-label">Author *</label>
                            <input type="text" class="form-control @error('author') is-invalid @enderror" 
                                   id="author" name="author" value="{{ old('author', 'Fashion Editor') }}" 
                                   placeholder="Author name">
                            @error('author')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Read Time -->
                        <div class="mb-3">
                            <label for="read_time" class="form-label">Read Time (minutes) *</label>
                            <input type="number" class="form-control @error('read_time') is-invalid @enderror" 
                                   id="read_time" name="read_time" value="{{ old('read_time', 5) }}" 
                                   min="1" max="30">
                            @error('read_time')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Published Date -->
                        <div class="mb-3">
                            <label for="published_date" class="form-label">Published Date</label>
                            <input type="date" class="form-control" 
                                   id="published_date" name="published_date" 
                                   value="{{ old('published_date', date('Y-m-d')) }}">
                        </div>
                    </div>
                </div>

                <!-- SEO Tips -->
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">SEO Tips</h6>
                    </div>
                    <div class="card-body">
                        <ul class="small text-muted mb-0">
                            <li>Include "Australia" or city names in title</li>
                            <li>Use "streetwear", "fashion", "trends" keywords</li>
                            <li>Write compelling meta descriptions</li>
                            <li>Add relevant Australian fashion brands</li>
                            <li>Include statistics and data</li>
                            <li>Minimum 1000 words for better ranking</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="card mt-4">
            <div class="card-body">
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-save me-2"></i>Create Article
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="saveDraft()">
                        <i class="fas fa-file-alt me-2"></i>Save Draft
                    </button>
                    <a href="{{ route('admin.trends.news') }}" class="btn btn-outline-secondary">
                        Cancel
                    </a>
                </div>
            </div>
        </div>
    </form>
</div>

<script>
// Initialize TinyMCE
tinymce.init({
    selector: '#content',
    height: 500,
    menubar: false,
    plugins: [
        'advlist', 'autolink', 'lists', 'link', 'image', 'charmap', 'preview',
        'anchor', 'searchreplace', 'visualblocks', 'code', 'fullscreen',
        'insertdatetime', 'media', 'table', 'help', 'wordcount'
    ],
    toolbar: 'undo redo | blocks | bold italic forecolor | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | removeformat | help',
    content_style: 'body { font-family: -apple-system, BlinkMacSystemFont, San Francisco, Segoe UI, Roboto, Helvetica Neue, sans-serif; font-size: 14px; -webkit-font-smoothing: antialiased; }'
});

// Image preview functionality
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').style.display = 'block';
        };
        reader.readAsDataURL(file);
    }
});

// Character counters
function setupCharacterCounter(inputId, counterId, maxLength) {
    const input = document.getElementById(inputId);
    const counter = document.getElementById(counterId);
    
    function updateCounter() {
        const length = input.value.length;
        counter.textContent = length;
        counter.parentElement.classList.toggle('text-warning', length > maxLength * 0.8);
        counter.parentElement.classList.toggle('text-danger', length > maxLength * 0.95);
    }
    
    input.addEventListener('input', updateCounter);
    updateCounter();
}

// Initialize character counters
document.addEventListener('DOMContentLoaded', function() {
    setupCharacterCounter('excerpt', 'excerptCount', 500);
    setupCharacterCounter('meta_title', 'metaTitleCount', 255);
    setupCharacterCounter('meta_description', 'metaDescCount', 300);
    
    // Auto-generate meta title from title
    document.getElementById('title').addEventListener('input', function() {
        const metaTitle = document.getElementById('meta_title');
        if (!metaTitle.value) {
            metaTitle.value = this.value;
            setupCharacterCounter('meta_title', 'metaTitleCount', 255);
        }
    });
});

// Save draft functionality
function saveDraft() {
    // In a real application, this would save to localStorage or make an AJAX call
    alert('Draft saved to browser storage!');
}

// Auto-save every 2 minutes
setInterval(function() {
    if (document.getElementById('title').value || tinymce.get('content').getContent()) {
        saveDraft();
    }
}, 120000);
</script>
@endsection