@extends('admin.layouts.admin')

@section('title', 'Edit Style Prediction')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3">Edit Style Prediction</h1>
            <p class="text-muted">Update existing fashion trend prediction</p>
        </div>
        <div>
            <a href="{{ route('trends.predictions') }}#{{ $prediction['slug'] }}" class="btn btn-info me-2" target="_blank">
                <i class="fas fa-eye me-2"></i>View Live
            </a>
            <a href="{{ route('admin.trends.predictions') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Back to Predictions
            </a>
        </div>
    </div>

    <!-- Form -->
    <div class="row">
        <div class="col-lg-8">
            <form action="{{ route('admin.trends.update-predictions', $prediction['slug']) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Prediction Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Title -->
                        <div class="mb-3">
                            <label for="title" class="form-label">Prediction Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title', $prediction['title']) }}" 
                                   placeholder="e.g., Smart Fabric Integration Revolution">
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="mb-3">
                            <label for="description" class="form-label">Description *</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="4" 
                                      placeholder="Describe the prediction and its potential impact...">{{ old('description', $prediction['description']) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Current Image -->
                        <div class="mb-3">
                            <label class="form-label">Current Prediction Image</label>
                            <div class="mb-2">
                                <img src="{{ \App\Helpers\ImageHelper::getImageUrl($prediction['image']) }}" 
                                     class="img-thumbnail" style="max-width: 300px;" 
                                     alt="{{ $prediction['title'] }}">
                            </div>
                        </div>

                        <!-- Image Upload -->
                        <div class="mb-3">
                            <label for="image" class="form-label">Update Prediction Image</label>
                            <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <div class="form-text">
                                Upload a new image to replace the current one (JPEG, PNG, JPG, GIF). Recommended size: 800x600px
                            </div>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            
                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="previewImg" src="" class="img-thumbnail" style="max-width: 300px;">
                            </div>
                        </div>

                        <div class="row">
                            <!-- Category -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="category" class="form-label">Category *</label>
                                    <select class="form-select @error('category') is-invalid @enderror" 
                                            id="category" name="category">
                                        <option value="">Select Category</option>
                                        <option value="Technology" {{ old('category', $prediction['category']) == 'Technology' ? 'selected' : '' }}>
                                            💻 Technology
                                        </option>
                                        <option value="Sustainability" {{ old('category', $prediction['category']) == 'Sustainability' ? 'selected' : '' }}>
                                            🌱 Sustainability
                                        </option>
                                        <option value="Materials" {{ old('category', $prediction['category']) == 'Materials' ? 'selected' : '' }}>
                                            🧬 Materials
                                        </option>
                                        <option value="Design Trends" {{ old('category', $prediction['category']) == 'Design Trends' ? 'selected' : '' }}>
                                            🎨 Design Trends
                                        </option>
                                    </select>
                                    @error('category')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Year -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="year" class="form-label">Target Year *</label>
                                    <input type="number" class="form-control @error('year') is-invalid @enderror" 
                                           id="year" name="year" value="{{ old('year', $prediction['year']) }}" 
                                           min="2024" max="2030">
                                    @error('year')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <!-- Confidence Level -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="confidence" class="form-label">Confidence Level *</label>
                                    <div class="row">
                                        <div class="col-8">
                                            <input type="range" class="form-range" id="confidenceRange" 
                                                   min="0" max="100" value="{{ old('confidence', $prediction['confidence']) }}" 
                                                   oninput="updateConfidenceValue(this.value)">
                                        </div>
                                        <div class="col-4">
                                            <div class="input-group">
                                                <input type="number" class="form-control @error('confidence') is-invalid @enderror" 
                                                       id="confidence" name="confidence" min="0" max="100" 
                                                       value="{{ old('confidence', $prediction['confidence']) }}" 
                                                       oninput="updateConfidenceRange(this.value)">
                                                <span class="input-group-text">%</span>
                                            </div>
                                        </div>
                                    </div>
                                    @error('confidence')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Impact Level -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="impact_level" class="form-label">Impact Level *</label>
                                    <select class="form-select @error('impact_level') is-invalid @enderror" 
                                            id="impact_level" name="impact_level">
                                        <option value="">Select Impact Level</option>
                                        <option value="Revolutionary" {{ old('impact_level', $prediction['impact_level']) == 'Revolutionary' ? 'selected' : '' }}>
                                            🚀 Revolutionary
                                        </option>
                                        <option value="Significant" {{ old('impact_level', $prediction['impact_level']) == 'Significant' ? 'selected' : '' }}>
                                            ⭐ Significant
                                        </option>
                                        <option value="Moderate" {{ old('impact_level', $prediction['impact_level']) == 'Moderate' ? 'selected' : '' }}>
                                            📊 Moderate
                                        </option>
                                    </select>
                                    @error('impact_level')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Timeline -->
                        <div class="mb-3">
                            <label for="timeline" class="form-label">Timeline *</label>
                            <input type="text" class="form-control @error('timeline') is-invalid @enderror" 
                                   id="timeline" name="timeline" value="{{ old('timeline', $prediction['timeline']) }}" 
                                   placeholder="e.g., Next 12-18 Months, Within 2 Years">
                            @error('timeline')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="card mt-3">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Update Prediction
                                </button>
                                <a href="{{ route('admin.trends.predictions') }}" class="btn btn-secondary">
                                    Cancel
                                </a>
                            </div>
                            
                            <button type="button" class="btn btn-outline-danger" 
                                    onclick="confirmDelete('{{ $prediction['slug'] }}', '{{ $prediction['title'] }}')">
                                <i class="fas fa-trash me-2"></i>Delete Prediction
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Preview Card -->
        <div class="col-lg-4">
            <div class="card sticky-top" style="top: 20px;">
                <div class="card-header">
                    <h6 class="mb-0">Prediction Preview</h6>
                </div>
                <div class="card-body">
                    <div id="previewCard">
                        <div class="preview-image mb-3" style="height: 200px; border-radius: 8px; overflow: hidden;">
                            <img src="{{ \App\Helpers\ImageHelper::getImageUrl($prediction['image']) }}" 
                                 style="width: 100%; height: 100%; object-fit: cover;" 
                                 alt="{{ $prediction['title'] }}" id="previewImage">
                        </div>
                        <div class="preview-badges mb-2">
                            <span class="badge bg-primary" id="previewCategory">{{ $prediction['category'] }}</span>
                            <span class="badge bg-info" id="previewYear">{{ $prediction['year'] }}</span>
                            <span class="badge bg-success" id="previewConfidence">{{ $prediction['confidence'] }}%</span>
                        </div>
                        <h6 id="previewTitle">{{ $prediction['title'] }}</h6>
                        <p id="previewDescription" class="small">{{ $prediction['description'] }}</p>
                        
                        <div class="mt-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <small class="fw-bold">Impact Level:</small>
                                <span class="badge bg-warning text-dark" id="previewImpact">{{ $prediction['impact_level'] }}</span>
                            </div>
                            <div class="d-flex justify-content-between align-items-center">
                                <small class="fw-bold">Timeline:</small>
                                <small class="text-muted" id="previewTimeline">{{ $prediction['timeline'] }}</small>
                            </div>
                        </div>

                        <!-- Impact Areas -->
                        <div class="mt-3">
                            <h6 class="small fw-bold">Impact Areas:</h6>
                            <div class="d-flex flex-wrap gap-1">
                                @foreach($prediction['impact_areas'] as $area)
                                    <span class="badge bg-light text-dark">{{ $area }}</span>
                                @endforeach
                            </div>
                        </div>
                    </div>
                    
                    <!-- Prediction Stats -->
                    <div class="border-top pt-3 mt-3">
                        <h6 class="small fw-bold mb-2">Prediction Statistics</h6>
                        <div class="row text-center">
                            <div class="col-6">
                                <div class="small text-muted">Created</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($prediction['created_at'])) }}</div>
                            </div>
                            <div class="col-6">
                                <div class="small text-muted">Updated</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($prediction['updated_at'])) }}</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this style prediction?</p>
                <p class="text-muted small">This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete Prediction</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Image preview functionality
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').style.display = 'block';
            
            // Update live preview
            document.getElementById('previewImage').src = e.target.result;
        };
        reader.readAsDataURL(file);
    }
});

// Confidence synchronization
function updateConfidenceValue(value) {
    document.getElementById('confidence').value = value;
    document.getElementById('previewConfidence').textContent = value + '%';
}

function updateConfidenceRange(value) {
    document.getElementById('confidenceRange').value = value;
    document.getElementById('previewConfidence').textContent = value + '%';
}

// Live preview updates
function updatePreview() {
    // Update title
    const title = document.getElementById('title').value || '{{ $prediction["title"] }}';
    document.getElementById('previewTitle').textContent = title;
    
    // Update description
    const description = document.getElementById('description').value || '{{ $prediction["description"] }}';
    document.getElementById('previewDescription').textContent = description.substring(0, 100) + (description.length > 100 ? '...' : '');
    
    // Update category
    const category = document.getElementById('category').value || '{{ $prediction["category"] }}';
    document.getElementById('previewCategory').textContent = category;
    
    // Update year
    const year = document.getElementById('year').value || '{{ $prediction["year"] }}';
    document.getElementById('previewYear').textContent = year;
    
    // Update impact level
    const impact = document.getElementById('impact_level').value || '{{ $prediction["impact_level"] }}';
    document.getElementById('previewImpact').textContent = impact;
    
    // Update timeline
    const timeline = document.getElementById('timeline').value || '{{ $prediction["timeline"] }}';
    document.getElementById('previewTimeline').textContent = timeline;
}

// Add event listeners for live preview
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('title').addEventListener('input', updatePreview);
    document.getElementById('description').addEventListener('input', updatePreview);
    document.getElementById('category').addEventListener('change', updatePreview);
    document.getElementById('year').addEventListener('input', updatePreview);
    document.getElementById('impact_level').addEventListener('change', updatePreview);
    document.getElementById('timeline').addEventListener('input', updatePreview);
});

// Delete confirmation
function confirmDelete(slug, title) {
    document.getElementById('deleteForm').action = `/admin/trends/predictions/${slug}`;
    new bootstrap.Modal(document.getElementById('deleteModal')).show();
}
</script>
@endsection