@extends('admin.layouts.admin')

@section('title', 'Edit Seasonal Forecast')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3">Edit Seasonal Forecast</h1>
            <p class="text-muted">Update existing seasonal fashion forecast</p>
        </div>
        <div>
            <a href="{{ route('trends.seasonal') }}#{{ $forecast['slug'] }}" class="btn btn-info me-2" target="_blank">
                <i class="fas fa-eye me-2"></i>View Live
            </a>
            <a href="{{ route('admin.trends.seasonal') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Back to Seasonal Forecasts
            </a>
        </div>
    </div>

    <!-- Form -->
    <div class="row">
        <div class="col-lg-8">
            <form action="{{ route('admin.trends.update-seasonal', $forecast['slug']) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Forecast Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Season -->
                        <div class="mb-3">
                            <label for="season" class="form-label">Season *</label>
                            <select class="form-select @error('season') is-invalid @enderror" 
                                    id="season" name="season">
                                <option value="">Select Season</option>
                                <option value="Spring" {{ old('season', $forecast['season']) == 'Spring' ? 'selected' : '' }}>
                                    🌸 Spring
                                </option>
                                <option value="Summer" {{ old('season', $forecast['season']) == 'Summer' ? 'selected' : '' }}>
                                    ☀️ Summer
                                </option>
                                <option value="Autumn" {{ old('season', $forecast['season']) == 'Autumn' ? 'selected' : '' }}>
                                    🍂 Autumn
                                </option>
                                <option value="Winter" {{ old('season', $forecast['season']) == 'Winter' ? 'selected' : '' }}>
                                    ❄️ Winter
                                </option>
                            </select>
                            @error('season')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Year -->
                        <div class="mb-3">
                            <label for="year" class="form-label">Year *</label>
                            <input type="number" class="form-control @error('year') is-invalid @enderror" 
                                   id="year" name="year" value="{{ old('year', $forecast['year']) }}" 
                                   min="2024" max="2030">
                            @error('year')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Title -->
                        <div class="mb-3">
                            <label for="title" class="form-label">Forecast Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title', $forecast['title']) }}" 
                                   placeholder="e.g., Fresh Beginnings: Spring Streetwear Trends">
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="mb-3">
                            <label for="description" class="form-label">Description *</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="4" 
                                      placeholder="Describe the seasonal forecast and key trends...">{{ old('description', $forecast['description']) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Current Image -->
                        <div class="mb-3">
                            <label class="form-label">Current Forecast Image</label>
                            <div class="mb-2">
                                <img src="{{ \App\Helpers\ImageHelper::getImageUrl($forecast['image']) }}" 
                                     class="img-thumbnail" style="max-width: 300px;" 
                                     alt="{{ $forecast['title'] }}">
                            </div>
                        </div>

                        <!-- Image Upload -->
                        <div class="mb-3">
                            <label for="image" class="form-label">Update Forecast Image</label>
                            <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <div class="form-text">
                                Upload a new image to replace the current one (JPEG, PNG, JPG, GIF). Recommended size: 800x600px
                            </div>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            
                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="previewImg" src="" class="img-thumbnail" style="max-width: 300px;">
                            </div>
                        </div>

                        <!-- Status -->
                        <div class="mb-3">
                            <label for="status" class="form-label">Status *</label>
                            <select class="form-select @error('status') is-invalid @enderror" 
                                    id="status" name="status">
                                <option value="active" {{ old('status', $forecast['status']) == 'active' ? 'selected' : '' }}>
                                    ✅ Active
                                </option>
                                <option value="draft" {{ old('status', $forecast['status']) == 'draft' ? 'selected' : '' }}>
                                    📝 Draft
                                </option>
                            </select>
                            @error('status')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="card mt-3">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Update Forecast
                                </button>
                                <a href="{{ route('admin.trends.seasonal') }}" class="btn btn-secondary">
                                    Cancel
                                </a>
                            </div>
                            
                            <button type="button" class="btn btn-outline-danger" 
                                    onclick="confirmDelete('{{ $forecast['slug'] }}', '{{ $forecast['title'] }}')">
                                <i class="fas fa-trash me-2"></i>Delete Forecast
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Preview Card -->
        <div class="col-lg-4">
            <div class="card sticky-top" style="top: 20px;">
                <div class="card-header">
                    <h6 class="mb-0">Forecast Preview</h6>
                </div>
                <div class="card-body">
                    <div id="previewCard">
                        <div class="preview-image mb-3" style="height: 200px; border-radius: 8px; overflow: hidden;">
                            <img src="{{ \App\Helpers\ImageHelper::getImageUrl($forecast['image']) }}" 
                                 style="width: 100%; height: 100%; object-fit: cover;" 
                                 alt="{{ $forecast['title'] }}" id="previewImage">
                        </div>
                        <div class="preview-badges mb-2">
                            <span class="badge bg-info" id="previewSeason">{{ $forecast['season'] }}</span>
                            <span class="badge bg-secondary" id="previewYear">{{ $forecast['year'] }}</span>
                            <span class="badge bg-success" id="previewStatus">{{ ucfirst($forecast['status']) }}</span>
                        </div>
                        <h6 id="previewTitle">{{ $forecast['title'] }}</h6>
                        <p id="previewDescription" class="small">{{ $forecast['description'] }}</p>
                        
                        <!-- Key Trends -->
                        <div class="mt-3">
                            <h6 class="small fw-bold">Key Trends:</h6>
                            <div class="d-flex flex-wrap gap-1">
                                @foreach($forecast['key_trends'] as $trend)
                                    <span class="badge bg-light text-dark">{{ $trend }}</span>
                                @endforeach
                            </div>
                        </div>

                        <!-- Color Palette -->
                        <div class="mt-3">
                            <h6 class="small fw-bold">Color Palette:</h6>
                            <div class="d-flex flex-wrap gap-1">
                                @foreach($forecast['color_palette'] as $color)
                                    <span class="badge bg-warning text-dark">{{ $color }}</span>
                                @endforeach
                            </div>
                        </div>
                    </div>
                    
                    <!-- Forecast Stats -->
                    <div class="border-top pt-3 mt-3">
                        <h6 class="small fw-bold mb-2">Forecast Statistics</h6>
                        <div class="row text-center">
                            <div class="col-6">
                                <div class="small text-muted">Created</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($forecast['created_at'])) }}</div>
                            </div>
                            <div class="col-6">
                                <div class="small text-muted">Updated</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($forecast['updated_at'])) }}</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this seasonal forecast?</p>
                <p class="text-muted small">This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete Forecast</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Image preview functionality
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').style.display = 'block';
            
            // Update live preview
            document.getElementById('previewImage').src = e.target.result;
        };
        reader.readAsDataURL(file);
    }
});

// Live preview updates
function updatePreview() {
    // Update title
    const title = document.getElementById('title').value || '{{ $forecast["title"] }}';
    document.getElementById('previewTitle').textContent = title;
    
    // Update description
    const description = document.getElementById('description').value || '{{ $forecast["description"] }}';
    document.getElementById('previewDescription').textContent = description.substring(0, 150) + (description.length > 150 ? '...' : '');
    
    // Update season
    const season = document.getElementById('season').value || '{{ $forecast["season"] }}';
    document.getElementById('previewSeason').textContent = season;
    
    // Update year
    const year = document.getElementById('year').value || '{{ $forecast["year"] }}';
    document.getElementById('previewYear').textContent = year;
    
    // Update status
    const status = document.getElementById('status').value || '{{ $forecast["status"] }}';
    const statusBadge = document.getElementById('previewStatus');
    statusBadge.textContent = status.charAt(0).toUpperCase() + status.slice(1);
    statusBadge.className = `badge ${status === 'active' ? 'bg-success' : 'bg-secondary'}`;
}

// Add event listeners for live preview
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('title').addEventListener('input', updatePreview);
    document.getElementById('description').addEventListener('input', updatePreview);
    document.getElementById('season').addEventListener('change', updatePreview);
    document.getElementById('year').addEventListener('input', updatePreview);
    document.getElementById('status').addEventListener('change', updatePreview);
});

// Delete confirmation
function confirmDelete(slug, title) {
    document.getElementById('deleteForm').action = `/admin/trends/seasonal/${slug}`;
    new bootstrap.Modal(document.getElementById('deleteModal')).show();
}
</script>
@endsection