@extends('admin.layouts.admin')

@section('title', 'Edit Weekly Trend')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3">Edit Weekly Trend</h1>
            <p class="text-muted">Update existing weekly trend information</p>
        </div>
        <div>
            <a href="{{ route('trends.weekly') }}" class="btn btn-info me-2" target="_blank">
                <i class="fas fa-eye me-2"></i>View Live
            </a>
            <a href="{{ route('admin.trends.weekly') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Back to Weekly Trends
            </a>
        </div>
    </div>

    <!-- Form -->
    <div class="row">
        <div class="col-lg-8">
            <form action="{{ route('admin.trends.update-weekly', $trend['slug']) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Trend Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Title -->
                        <div class="mb-3">
                            <label for="title" class="form-label">Trend Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title', $trend['title']) }}" 
                                   placeholder="e.g., Sustainable Streetwear Revolution">
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="mb-3">
                            <label for="description" class="form-label">Description *</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="3" 
                                      placeholder="Describe the trend and its impact on fashion...">{{ old('description', $trend['description']) }}</textarea>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Current Image -->
                        <div class="mb-3">
                            <label class="form-label">Current Trend Image</label>
                            <div class="mb-2">
                                <img src="{{ \App\Helpers\ImageHelper::getImageUrl($trend['image']) }}" 
                                     class="img-thumbnail" style="max-width: 200px;" 
                                     alt="{{ $trend['title'] }}">
                            </div>
                        </div>

                        <!-- Image Upload -->
                        <div class="mb-3">
                            <label for="image" class="form-label">Update Trend Image</label>
                            <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <div class="form-text">
                                Upload a new image to replace the current one (JPEG, PNG, JPG, GIF). Recommended size: 800x600px
                            </div>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            
                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="previewImg" src="" class="img-thumbnail" style="max-width: 200px;">
                            </div>
                        </div>

                        <!-- Trend Level -->
                        <div class="mb-3">
                            <label for="trend_level" class="form-label">Trend Level *</label>
                            <select class="form-select @error('trend_level') is-invalid @enderror" 
                                    id="trend_level" name="trend_level">
                                <option value="">Select Trend Level</option>
                                <option value="Rising Fast" {{ old('trend_level', $trend['trend_level']) == 'Rising Fast' ? 'selected' : '' }}>
                                    🚀 Rising Fast
                                </option>
                                <option value="Mainstream" {{ old('trend_level', $trend['trend_level']) == 'Mainstream' ? 'selected' : '' }}>
                                    ⭐ Mainstream
                                </option>
                                <option value="Trending" {{ old('trend_level', $trend['trend_level']) == 'Trending' ? 'selected' : '' }}>
                                    📈 Trending
                                </option>
                            </select>
                            @error('trend_level')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Popularity Score -->
                        <div class="mb-3">
                            <label for="popularity" class="form-label">Popularity Score *</label>
                            <div class="row">
                                <div class="col-md-6">
                                    <input type="range" class="form-range" id="popularityRange" 
                                           min="0" max="100" value="{{ old('popularity', $trend['popularity']) }}" 
                                           oninput="updatePopularityValue(this.value)">
                                </div>
                                <div class="col-md-6">
                                    <div class="input-group">
                                        <input type="number" class="form-control @error('popularity') is-invalid @enderror" 
                                               id="popularity" name="popularity" min="0" max="100" 
                                               value="{{ old('popularity', $trend['popularity']) }}" 
                                               oninput="updatePopularityRange(this.value)">
                                        <span class="input-group-text">%</span>
                                    </div>
                                </div>
                            </div>
                            @error('popularity')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Key Pieces -->
                        <div class="mb-3">
                            <label class="form-label">Key Pieces *</label>
                            <div id="keyPiecesContainer">
                                @foreach($trend['key_pieces'] as $index => $piece)
                                <div class="input-group mb-2">
                                    <input type="text" class="form-control" name="key_pieces[]" 
                                           placeholder="e.g., Heritage Hoodies" value="{{ old('key_pieces.'.$index, $piece) }}"
                                           oninput="updatePreview()">
                                    <button class="btn btn-outline-danger" type="button" onclick="removeKeyPiece(this)">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                @endforeach
                                @if(old('key_pieces') && count(old('key_pieces')) > count($trend['key_pieces']))
                                    @foreach(array_slice(old('key_pieces'), count($trend['key_pieces'])) as $piece)
                                    <div class="input-group mb-2">
                                        <input type="text" class="form-control" name="key_pieces[]" 
                                               placeholder="e.g., Cultural Print Tees" value="{{ $piece }}"
                                               oninput="updatePreview()">
                                        <button class="btn btn-outline-danger" type="button" onclick="removeKeyPiece(this)">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                    @endforeach
                                @endif
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="addKeyPiece()">
                                <i class="fas fa-plus me-1"></i>Add Another Piece
                            </button>
                            @error('key_pieces')
                                <div class="text-danger small mt-1">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="card mt-3">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Update Trend
                                </button>
                                <a href="{{ route('admin.trends.weekly') }}" class="btn btn-secondary">
                                    Cancel
                                </a>
                            </div>
                            
                            <button type="button" class="btn btn-outline-danger" 
                                    onclick="confirmDelete('{{ $trend['slug'] }}', '{{ $trend['title'] }}')">
                                <i class="fas fa-trash me-2"></i>Delete Trend
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Preview Card -->
        <div class="col-lg-4">
            <div class="card sticky-top" style="top: 20px;">
                <div class="card-header">
                    <h6 class="mb-0">Live Preview</h6>
                </div>
                <div class="card-body">
                    <div id="previewCard" class="trend-preview">
                        <div class="preview-image mb-3" style="height: 200px; border-radius: 8px; overflow: hidden;">
                            <img src="{{ \App\Helpers\ImageHelper::getImageUrl($trend['image']) }}" 
                                 style="width: 100%; height: 100%; object-fit: cover;" 
                                 alt="{{ $trend['title'] }}" id="previewImage">
                        </div>
                        <div class="preview-badges mb-2">
                            <span class="badge {{ getTrendLevelClass($trend['trend_level']) }}" id="previewLevel">{{ $trend['trend_level'] }}</span>
                            <span class="badge bg-info" id="previewPopularity">{{ $trend['popularity'] }}%</span>
                        </div>
                        <h6 id="previewTitle">{{ $trend['title'] }}</h6>
                        <p id="previewDescription" class="small">{{ $trend['description'] }}</p>
                        <div id="previewKeyPieces" class="mt-2">
                            @foreach($trend['key_pieces'] as $piece)
                                <span class="badge bg-light text-dark me-1 mb-1">{{ $piece }}</span>
                            @endforeach
                        </div>
                    </div>
                    
                    <!-- Trend Stats -->
                    <div class="border-top pt-3 mt-3">
                        <h6 class="small fw-bold mb-2">Trend Statistics</h6>
                        <div class="row text-center">
                            <div class="col-6">
                                <div class="small text-muted">Created</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($trend['created_at'])) }}</div>
                            </div>
                            <div class="col-6">
                                <div class="small text-muted">Updated</div>
                                <div class="fw-bold">{{ date('M j, Y', strtotime($trend['updated_at'])) }}</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this trend?</p>
                <p class="text-muted small">This action cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" class="d-inline">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete Trend</button>
                </form>
            </div>
        </div>
    </div>
</div>

@php
function getTrendLevelClass($level) {
    switch($level) {
        case 'Rising Fast': return 'bg-danger';
        case 'Mainstream': return 'bg-success';
        case 'Trending': return 'bg-warning text-dark';
        default: return 'bg-secondary';
    }
}
@endphp

<script>
// Image preview functionality
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').style.display = 'block';
            
            // Update live preview
            document.getElementById('previewImage').src = e.target.result;
        };
        reader.readAsDataURL(file);
    }
});

// Popularity score synchronization
function updatePopularityValue(value) {
    document.getElementById('popularity').value = value;
    document.getElementById('previewPopularity').textContent = value + '%';
}

function updatePopularityRange(value) {
    document.getElementById('popularityRange').value = value;
    document.getElementById('previewPopularity').textContent = value + '%';
}

// Key pieces management
function addKeyPiece() {
    const container = document.getElementById('keyPiecesContainer');
    const newDiv = document.createElement('div');
    newDiv.className = 'input-group mb-2';
    newDiv.innerHTML = `
        <input type="text" class="form-control" name="key_pieces[]" placeholder="e.g., Sustainable Materials" oninput="updatePreview()">
        <button class="btn btn-outline-danger" type="button" onclick="removeKeyPiece(this)">
            <i class="fas fa-times"></i>
        </button>
    `;
    container.appendChild(newDiv);
    updatePreview();
}

function removeKeyPiece(button) {
    if (document.querySelectorAll('#keyPiecesContainer .input-group').length > 1) {
        button.parentElement.remove();
        updatePreview();
    }
}

// Live preview updates
function updatePreview() {
    // Update title
    const title = document.getElementById('title').value || '{{ $trend["title"] }}';
    document.getElementById('previewTitle').textContent = title;
    
    // Update description
    const description = document.getElementById('description').value || '{{ $trend["description"] }}';
    document.getElementById('previewDescription').textContent = description.substring(0, 100) + (description.length > 100 ? '...' : '');
    
    // Update trend level
    const level = document.getElementById('trend_level').value || '{{ $trend["trend_level"] }}';
    const levelBadge = document.getElementById('previewLevel');
    levelBadge.textContent = level;
    levelBadge.className = `badge ${getTrendLevelClass(level)}`;
    
    // Update key pieces
    const keyPieces = Array.from(document.querySelectorAll('input[name="key_pieces[]"]'))
                          .map(input => input.value)
                          .filter(value => value.trim() !== '');
    
    const keyPiecesContainer = document.getElementById('previewKeyPieces');
    keyPiecesContainer.innerHTML = keyPieces.map(piece => 
        `<span class="badge bg-light text-dark me-1 mb-1">${piece}</span>`
    ).join('');
}

function getTrendLevelClass(level) {
    switch(level) {
        case 'Rising Fast': return 'bg-danger';
        case 'Mainstream': return 'bg-success';
        case 'Trending': return 'bg-warning text-dark';
        default: return 'bg-secondary';
    }
}

// Add event listeners for live preview
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('title').addEventListener('input', updatePreview);
    document.getElementById('description').addEventListener('input', updatePreview);
    document.getElementById('trend_level').addEventListener('change', updatePreview);
    
    // Add listeners to existing key piece inputs
    document.querySelectorAll('input[name="key_pieces[]"]').forEach(input => {
        input.addEventListener('input', updatePreview);
    });
});

// Delete confirmation
function confirmDelete(slug, title) {
    document.getElementById('deleteForm').action = `/admin/trends/weekly/${slug}`;
    new bootstrap.Modal(document.getElementById('deleteModal')).show();
}
</script>
@endsection