<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class ImageHelper
{
    /**
     * Get the URL for a stored image - Works on any hosting
     */
    public static function getImageUrl($path)
    {
        // Handle empty or null paths
        if (empty($path)) {
            return asset('images/placeholder-product.jpg');
        }

        // Clean the path
        $path = trim($path, '/');
        $path = str_replace('\\', '/', $path);

        // Method 1: Check if file exists in public/storage (symlink or copied files)
        if (file_exists(public_path('storage/' . $path))) {
            return asset('storage/' . $path);
        }

        // Method 2: For shared hosting - use direct serving route
        // This will work even without symlinks
        if (config('app.env') === 'production' || !file_exists(public_path('storage'))) {
            // Use our custom image serving route
            return route('serve.image', ['path' => base64_encode($path)]);
        }

        // Method 3: Local development with Storage facade
        if (Storage::disk('public')->exists($path)) {
            return asset('storage/' . $path);
        }

        // Return placeholder if image doesn't exist
        return asset('images/placeholder-product.jpg');
    }

    /**
     * Get multiple image URLs (for gallery images)
     */
    public static function getGalleryImages($jsonImages)
    {
        if (empty($jsonImages)) {
            return [];
        }

        // Handle JSON string or array
        $images = is_string($jsonImages) ? json_decode($jsonImages, true) : $jsonImages;
        
        if (!is_array($images)) {
            return [];
        }

        return array_map(function($image) {
            return self::getImageUrl($image);
        }, $images);
    }

    /**
     * Serve image directly from storage (for hosting without symlinks)
     */
    public static function serveImage($encodedPath)
    {
        try {
            $path = base64_decode($encodedPath);
            $path = trim($path, '/');
            
            // Security check - prevent directory traversal
            if (strpos($path, '..') !== false || strpos($path, './') !== false) {
                abort(403);
            }

            // Try different possible locations
            $possiblePaths = [
                storage_path('app/public/' . $path),
                storage_path('app/' . $path),
                public_path('storage/' . $path),
            ];

            foreach ($possiblePaths as $fullPath) {
                if (file_exists($fullPath) && is_file($fullPath)) {
                    $mimeType = mime_content_type($fullPath);
                    
                    // Serve the image with proper headers
                    return response()->file($fullPath, [
                        'Content-Type' => $mimeType,
                        'Cache-Control' => 'public, max-age=31536000',
                        'Pragma' => 'public',
                        'Expires' => gmdate('D, d M Y H:i:s', time() + 31536000) . ' GMT',
                    ]);
                }
            }

            // If image not found, serve placeholder
            $placeholderPath = public_path('images/placeholder-product.jpg');
            if (file_exists($placeholderPath)) {
                return response()->file($placeholderPath);
            }
            
            abort(404);
            
        } catch (\Exception $e) {
            // Log error and return placeholder
            \Log::error('Image serving error: ' . $e->getMessage());
            
            $placeholderPath = public_path('images/placeholder-product.jpg');
            if (file_exists($placeholderPath)) {
                return response()->file($placeholderPath);
            }
            
            abort(404);
        }
    }

    /**
     * Check if image exists in storage
     */
    public static function imageExists($path)
    {
        if (empty($path)) {
            return false;
        }

        $path = trim($path, '/');

        // Check multiple locations
        if (file_exists(public_path('storage/' . $path))) {
            return true;
        }

        if (file_exists(storage_path('app/public/' . $path))) {
            return true;
        }

        return Storage::disk('public')->exists($path);
    }

    /**
     * Get image URL with fallback
     */
    public static function getImageUrlWithFallback($path, $fallback = null)
    {
        if (self::imageExists($path)) {
            return self::getImageUrl($path);
        }

        if ($fallback && self::imageExists($fallback)) {
            return self::getImageUrl($fallback);
        }

        return asset('images/placeholder-product.jpg');
    }

    /**
     * Get the company favicon URL
     */
    public static function getFaviconUrl()
    {
        $companyInfo = \App\Models\CompanyInfo::getInfo();

        if (!empty($companyInfo->favicon)) {
            return self::getImageUrl($companyInfo->favicon);
        }

        return asset('images/E.png');
    }
}