<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Wishlist;
use App\Models\ShippingAddress;
use App\Models\Cart;
use App\Models\Category;
use App\Models\CompanyInfo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AccountController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'verified']);
    }

    public function dashboard()
    {
        $user = Auth::user();
        
        $stats = [
            'total_orders'   => Order::where('user_id', $user->id)->count(),
            'pending_orders' => Order::where('user_id', $user->id)->where('status', 'pending')->count(),
            'wishlist_items' => Wishlist::where('user_id', $user->id)->count(),
            'addresses'      => ShippingAddress::where('user_id', $user->id)->count(),
        ];
        
        $recentOrders = Order::where('user_id', $user->id)
            ->with(['shippingAddress'])
            ->latest()
            ->take(5)
            ->get();
        
        // Add tax rate and shipping zone information to recent orders and recalculate amounts
        foreach ($recentOrders as $order) {
            if ($order->shippingAddress) {
                // Get tax rate for this order's location
                if (class_exists(\App\Models\TaxRate::class)) {
                    $order->taxRateInfo = \App\Models\TaxRate::findByCity(
                        $order->shippingAddress->city,
                        $order->shippingAddress->state
                    );
                    
                    // Recalculate tax amount based on current admin rates
                    if ($order->taxRateInfo) {
                        $order->calculated_tax_amount = $order->taxRateInfo->calculateTax($order->subtotal);
                        $order->calculated_total_amount = $order->subtotal + $order->calculated_tax_amount + $order->shipping_amount - $order->discount_amount;
                    }
                }
                
                // Get shipping zone for this order's location
                if (class_exists(\App\Models\ShippingZone::class)) {
                    $order->shippingZoneInfo = \App\Models\ShippingZone::findByCity(
                        $order->shippingAddress->city,
                        $order->shippingAddress->state
                    );
                }
            }
        }
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.dashboard', compact('user', 'stats', 'recentOrders', 'categories', 'companyInfo'));
    }

    public function orders()
    {
        $orders = Order::where('user_id', Auth::id())
            ->with(['items', 'shippingAddress'])
            ->latest()
            ->paginate(10);
        
        // Add tax rate and shipping zone information to orders and recalculate amounts
        foreach ($orders as $order) {
            if ($order->shippingAddress) {
                // Get tax rate for this order's location
                if (class_exists(\App\Models\TaxRate::class)) {
                    $order->taxRateInfo = \App\Models\TaxRate::findByCity(
                        $order->shippingAddress->city,
                        $order->shippingAddress->state
                    );
                    
                    // Recalculate tax amount based on current admin rates
                    if ($order->taxRateInfo) {
                        $order->calculated_tax_amount = $order->taxRateInfo->calculateTax($order->subtotal);
                        $order->calculated_total_amount = $order->subtotal + $order->calculated_tax_amount + $order->shipping_amount - $order->discount_amount;
                    }
                }
                
                // Get shipping zone for this order's location
                if (class_exists(\App\Models\ShippingZone::class)) {
                    $order->shippingZoneInfo = \App\Models\ShippingZone::findByCity(
                        $order->shippingAddress->city,
                        $order->shippingAddress->state
                    );
                }
            }
        }
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.orders', compact('orders', 'categories', 'companyInfo'));
    }

    public function orderDetail($orderNumber)
    {
        $order = Order::where('order_number', $orderNumber)
            ->where('user_id', Auth::id())
            ->with(['items.product', 'shippingAddress'])
            ->firstOrFail();
        
        // Get tax rate details for the order's shipping location and recalculate amounts
        $taxRate = null;
        if ($order->shippingAddress) {
            $taxRate = \App\Models\TaxRate::findByCity(
                $order->shippingAddress->city,
                $order->shippingAddress->state
            );
            
            // Recalculate tax amount based on current admin rates
            if ($taxRate) {
                $order->calculated_tax_amount = $taxRate->calculateTax($order->subtotal);
                $order->calculated_total_amount = $order->subtotal + $order->calculated_tax_amount + $order->shipping_amount - $order->discount_amount;
            }
        }
        
        // Get shipping zone details for the order's shipping location  
        $shippingZone = null;
        if ($order->shippingAddress) {
            $shippingZone = \App\Models\ShippingZone::findByCity(
                $order->shippingAddress->city,
                $order->shippingAddress->state
            );
        }
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.order-detail', compact('order', 'categories', 'companyInfo', 'taxRate', 'shippingZone'));
    }

    public function wishlist()
    {
        $wishlistItems = Wishlist::where('user_id', Auth::id())
            ->with('product')
            ->paginate(12);
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.wishlist', compact('wishlistItems', 'categories', 'companyInfo'));
    }

    public function removeFromWishlist($id)
    {
        $wishlistItem = Wishlist::where('id', $id)
            ->where('user_id', Auth::id())
            ->firstOrFail();
        
        $wishlistItem->delete();
        
        return redirect()->route('account.wishlist')->with('success', 'Item removed from wishlist!');
    }

    public function addresses()
    {
        $addresses = ShippingAddress::where('user_id', Auth::id())
            ->latest()
            ->get();
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.addresses', compact('addresses', 'categories', 'companyInfo'));
    }

    public function addAddress(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address_line_1' => 'required|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'required|string|max:20',
            'is_default' => 'boolean'
        ]);

        // If this is set as default, unset other defaults
        if ($request->is_default) {
            ShippingAddress::where('user_id', Auth::id())
                ->update(['is_default' => false]);
        }

        $validated['user_id'] = Auth::id();
        ShippingAddress::create($validated);

        return redirect()->route('account.addresses')->with('success', 'Address added successfully!');
    }

    public function updateAddress(Request $request, $id)
    {
        $address = ShippingAddress::where('id', $id)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'address_line_1' => 'required|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'country' => 'required|string|max:100',
            'postal_code' => 'required|string|max:20',
            'is_default' => 'boolean'
        ]);

        // If this is set as default, unset other defaults
        if ($request->is_default && !$address->is_default) {
            ShippingAddress::where('user_id', Auth::id())
                ->where('id', '!=', $id)
                ->update(['is_default' => false]);
        }

        $address->update($validated);

        return redirect()->route('account.addresses')->with('success', 'Address updated successfully!');
    }

    public function deleteAddress($id)
    {
        $address = ShippingAddress::where('id', $id)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        $address->delete();

        return redirect()->route('account.addresses')->with('success', 'Address deleted successfully!');
    }

    public function profile()
    {
        $user = Auth::user();
        
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('account.profile', compact('user', 'categories', 'companyInfo'));
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'current_password' => 'nullable|required_with:new_password',
            'new_password' => 'nullable|min:8|confirmed|different:current_password',
        ]);

        // Update basic info
        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $user->phone = $validated['phone'] ?? null;

        // Update password if provided
        if ($request->filled('current_password')) {
            if (!Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => 'Current password is incorrect']);
            }

            $user->password = Hash::make($request->new_password);
        }

        $user->save();

        return redirect()->route('account.profile')->with('success', 'Profile updated successfully!');
    }

    public function deleteAccount(Request $request)
    {
        $request->validate([
            'password' => 'required',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'Password is incorrect']);
        }

        // Delete user's related data
        $user->orders()->delete();
        $user->wishlist()->delete();
        $user->carts()->delete();
        $user->shippingAddresses()->delete();
        
        // Delete the user
        $user->delete();

        Auth::logout();

        return redirect('/')->with('success', 'Your account has been deleted successfully.');
    }
}