<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BannerController extends Controller
{
    /**
     * Display a listing of banners.
     */
    public function index()
    {
        $banners = Banner::orderBy('position')
                         ->orderBy('sort_order')
                         ->paginate(10);
        
        return view('admin.banners.index', compact('banners'));
    }

    /**
     * Show the form for creating a new banner.
     */
    public function create()
    {
        return view('admin.banners.create');
    }

    /**
     * Store a newly created banner.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'slide_duration' => 'nullable|integer|min:3|max:10',
            'animation_type' => 'nullable|in:fade,slide',
            'auto_play' => 'nullable|boolean',
            'button_text' => 'nullable|string|max:50',
            'button_link' => 'nullable|string|max:255',
            'position' => 'required|in:hero,promo_left,promo_right,home_hero_slider,below_categories_slider,train_slider',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean'
        ]);

        // Handle single image upload (for backward compatibility)
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('banners', 'public');
            $validated['image'] = $path;
        }

        // Handle multiple images upload
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {
                $path = $file->store('banners', 'public');
                $imagesPaths[] = $path;
            }
        }

        // If no multiple images but single image exists, use single image
        if (empty($imagesPaths) && isset($validated['image'])) {
            $imagesPaths = [$validated['image']];
        }

        // If we have multiple images but no single image, set the first multiple image as the single image
        if (!empty($imagesPaths) && !isset($validated['image'])) {
            $validated['image'] = $imagesPaths[0];
        }

        $validated['images'] = !empty($imagesPaths) ? $imagesPaths : null;
        $validated['is_active'] = $request->has('is_active');
        $validated['auto_play'] = $request->has('auto_play');
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        $validated['slide_duration'] = $validated['slide_duration'] ?? 5;
        $validated['animation_type'] = $validated['animation_type'] ?? 'fade';

        Banner::create($validated);

        return redirect()->route('admin.banners.index')
                        ->with('success', 'Banner created successfully!');
    }

    /**
     * Show the form for editing the specified banner.
     */
    public function edit(Banner $banner)
    {
        return view('admin.banners.edit', compact('banner'));
    }

    /**
     * Update the specified banner.
     */
    public function update(Request $request, Banner $banner)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'images.*' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'slide_duration' => 'nullable|integer|min:3|max:10',
            'animation_type' => 'nullable|in:fade,slide',
            'auto_play' => 'nullable|boolean',
            'button_text' => 'nullable|string|max:50',
            'button_link' => 'nullable|string|max:255',
            'position' => 'required|in:hero,promo_left,promo_right,home_hero_slider,below_categories_slider,train_slider',
            'sort_order' => 'nullable|integer',
            'is_active' => 'boolean'
        ]);

        // Handle single image upload (for backward compatibility)
        if ($request->hasFile('image')) {
            // Delete old image
            if ($banner->image && Storage::disk('public')->exists($banner->image)) {
                Storage::disk('public')->delete($banner->image);
            }

            $path = $request->file('image')->store('banners', 'public');
            $validated['image'] = $path;
        }

        // Handle multiple images upload
        $imagesPaths = [];
        if ($request->hasFile('images')) {
            // Delete old images if new ones are uploaded
            if ($banner->images && is_array($banner->images)) {
                foreach ($banner->images as $oldImage) {
                    if (Storage::disk('public')->exists($oldImage)) {
                        Storage::disk('public')->delete($oldImage);
                    }
                }
            }

            foreach ($request->file('images') as $file) {
                $path = $file->store('banners', 'public');
                $imagesPaths[] = $path;
            }
        } else {
            // Keep existing images if no new ones uploaded
            $imagesPaths = $banner->images ?? [];
        }

        // If no multiple images but single image exists, use single image
        if (empty($imagesPaths) && isset($validated['image'])) {
            $imagesPaths = [$validated['image']];
        }

        // If we have multiple images but no single image, set the first multiple image as the single image
        if (!empty($imagesPaths) && !isset($validated['image'])) {
            $validated['image'] = $imagesPaths[0];
        }

        $validated['images'] = !empty($imagesPaths) ? $imagesPaths : null;
        $validated['is_active'] = $request->has('is_active');
        $validated['auto_play'] = $request->has('auto_play');
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        $validated['slide_duration'] = $validated['slide_duration'] ?? 5;
        $validated['animation_type'] = $validated['animation_type'] ?? 'fade';

        $banner->update($validated);

        return redirect()->route('admin.banners.index')
                        ->with('success', 'Banner updated successfully!');
    }

    /**
     * Remove the specified banner.
     */
    public function destroy(Banner $banner)
    {
        // Delete single image file
        if ($banner->image && Storage::disk('public')->exists($banner->image)) {
            Storage::disk('public')->delete($banner->image);
        }

        // Delete multiple images
        if ($banner->images && is_array($banner->images)) {
            foreach ($banner->images as $image) {
                if (Storage::disk('public')->exists($image)) {
                    Storage::disk('public')->delete($image);
                }
            }
        }

        $banner->delete();

        return redirect()->route('admin.banners.index')
                        ->with('success', 'Banner deleted successfully!');
    }

    /**
     * Toggle active status.
     */
    public function toggleActive(Banner $banner)
    {
        $banner->update(['is_active' => !$banner->is_active]);

        return redirect()->route('admin.banners.index')
                        ->with('success', 'Banner status updated!');
    }
}