<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CommunityPost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CommunityController extends Controller
{
    /**
     * Display a listing of community posts.
     */
    public function index(Request $request)
    {
        $query = CommunityPost::query();

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search by name or location
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('location', 'like', "%{$search}%")
                  ->orWhere('caption', 'like', "%{$search}%");
            });
        }

        $communityPosts = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.community.index', compact('communityPosts'));
    }

    /**
     * Show the form for creating a new community post.
     */
    public function create()
    {
        return view('admin.community.create');
    }

    /**
     * Store a newly created community post in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'caption' => 'required|string|max:1000',
            'products_featured' => 'nullable|string|max:500',
            'instagram_handle' => 'nullable|string|max:255',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:10240',
            'status' => 'required|in:pending,approved,rejected',
            'featured' => 'boolean',
            'likes' => 'numeric|min:0'
        ]);

        $communityPost = new CommunityPost($request->except('image'));

        // Handle image upload
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = 'community_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('images/community'), $filename);
            $communityPost->image_path = 'community/' . $filename;
        }

        $communityPost->featured = $request->boolean('featured');
        $communityPost->likes = $request->likes ?? 0;
        $communityPost->consent_gallery = true; // Admin created posts have consent
        $communityPost->consent_social = $request->boolean('consent_social');
        $communityPost->consent_marketing = $request->boolean('consent_marketing');

        if ($request->status === 'approved') {
            $communityPost->approved_at = now();
        }

        $communityPost->save();

        return redirect()->route('admin.community.index')
                        ->with('success', 'Community post created successfully.');
    }

    /**
     * Display the specified community post.
     */
    public function show(CommunityPost $community)
    {
        return view('admin.community.show', compact('community'));
    }

    /**
     * Show the form for editing the specified community post.
     */
    public function edit(CommunityPost $community)
    {
        return view('admin.community.edit', compact('community'));
    }

    /**
     * Update the specified community post in storage.
     */
    public function update(Request $request, CommunityPost $community)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'caption' => 'required|string|max:1000',
            'products_featured' => 'nullable|string|max:500',
            'instagram_handle' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:10240',
            'status' => 'required|in:pending,approved,rejected',
            'featured' => 'boolean',
            'likes' => 'numeric|min:0'
        ]);

        // Handle image upload if new image is provided
        if ($request->hasFile('image')) {
            // Delete old image
            if ($community->image_path && file_exists(public_path('images/' . $community->image_path))) {
                unlink(public_path('images/' . $community->image_path));
            }

            $image = $request->file('image');
            $filename = 'community_' . time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('images/community'), $filename);
            $community->image_path = 'community/' . $filename;
        }

        $community->fill($request->except('image'));
        $community->featured = $request->boolean('featured');
        $community->likes = $request->likes ?? 0;
        $community->consent_social = $request->boolean('consent_social');
        $community->consent_marketing = $request->boolean('consent_marketing');

        // Update approved_at when status changes to approved
        if ($request->status === 'approved' && $community->status !== 'approved') {
            $community->approved_at = now();
        } elseif ($request->status !== 'approved') {
            $community->approved_at = null;
        }

        $community->save();

        return redirect()->route('admin.community.index')
                        ->with('success', 'Community post updated successfully.');
    }

    /**
     * Remove the specified community post from storage.
     */
    public function destroy(CommunityPost $community)
    {
        // Delete image file
        if ($community->image_path && file_exists(public_path('images/' . $community->image_path))) {
            unlink(public_path('images/' . $community->image_path));
        }

        $community->delete();

        return redirect()->route('admin.community.index')
                        ->with('success', 'Community post deleted successfully.');
    }

    /**
     * Toggle featured status of community post.
     */
    public function toggleFeatured(CommunityPost $community)
    {
        $community->featured = !$community->featured;
        $community->save();

        $status = $community->featured ? 'featured' : 'unfeatured';
        
        return response()->json([
            'success' => true,
            'message' => "Community post {$status} successfully.",
            'featured' => $community->featured
        ]);
    }

    /**
     * Approve community post.
     */
    public function approve(CommunityPost $community)
    {
        $community->update([
            'status' => 'approved',
            'approved_at' => now()
        ]);

        return redirect()->back()->with('success', 'Community post approved successfully.');
    }

    /**
     * Reject community post.
     */
    public function reject(CommunityPost $community)
    {
        $community->update([
            'status' => 'rejected',
            'approved_at' => null
        ]);

        return redirect()->back()->with('success', 'Community post rejected.');
    }

    /**
     * Bulk actions for community posts.
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:approve,reject,delete,toggle_featured',
            'posts' => 'required|array',
            'posts.*' => 'exists:community_posts,id'
        ]);

        $posts = CommunityPost::whereIn('id', $request->posts);

        switch ($request->action) {
            case 'approve':
                $posts->update([
                    'status' => 'approved',
                    'approved_at' => now()
                ]);
                $message = 'Selected posts approved successfully.';
                break;

            case 'reject':
                $posts->update([
                    'status' => 'rejected',
                    'approved_at' => null
                ]);
                $message = 'Selected posts rejected successfully.';
                break;

            case 'delete':
                foreach ($posts->get() as $post) {
                    if ($post->image_path && file_exists(public_path('images/' . $post->image_path))) {
                        unlink(public_path('images/' . $post->image_path));
                    }
                }
                $posts->delete();
                $message = 'Selected posts deleted successfully.';
                break;

            case 'toggle_featured':
                foreach ($posts->get() as $post) {
                    $post->update(['featured' => !$post->featured]);
                }
                $message = 'Featured status toggled for selected posts.';
                break;

            default:
                $message = 'Action completed.';
        }

        return redirect()->back()->with('success', $message);
    }
}