<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ProductController extends Controller
{
    /**
     * Display a listing of products.
     */
    public function index()
    {
        $products = Product::with('category')
            ->latest()
            ->paginate(10);
        
        return view('admin.products.index', compact('products'));
    }

    /**
     * Show the form for creating a new product.
     */
    public function create()
    {
        $categories = Category::where('is_active', true)
            ->orderBy('name')
            ->get();
        
        return view('admin.products.create', compact('categories'));
    }

    /**
     * Store a newly created product in storage.
     */
    public function store(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string|max:500',
            'category_id' => 'required|exists:categories,id',
            'sku' => 'nullable|string|unique:products,sku',
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'cost_price' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'hover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'gallery_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'sizes' => 'nullable|array',
            'colors' => 'nullable|array',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
        ]);

        DB::beginTransaction();
        
        try {
            // Generate slug
            $validated['slug'] = Str::slug($validated['name']);
            
            // Make slug unique if needed
            $count = Product::where('slug', 'like', $validated['slug'] . '%')->count();
            if ($count > 0) {
                $validated['slug'] = $validated['slug'] . '-' . ($count + 1);
            }
            
            // Handle checkboxes
            $validated['is_featured'] = $request->has('is_featured');
            $validated['is_active'] = $request->has('is_active');
            
            // Handle featured image upload
            if ($request->hasFile('featured_image')) {
                $featuredImage = $request->file('featured_image');
                $filename = time() . '_' . Str::slug($validated['name']) . '.' . $featuredImage->getClientOriginalExtension();
                $path = $featuredImage->storeAs('products', $filename, 'public');
                $validated['featured_image'] = $path;
            }

            // Handle hover image upload
            if ($request->hasFile('hover_image')) {
                $hoverImage = $request->file('hover_image');
                $filename = time() . '_hover_' . Str::slug($validated['name']) . '.' . $hoverImage->getClientOriginalExtension();
                $path = $hoverImage->storeAs('products', $filename, 'public');
                $validated['hover_image'] = $path;
            }

            // Handle gallery images
            $galleryPaths = [];
            if ($request->hasFile('gallery_images')) {
                foreach ($request->file('gallery_images') as $index => $image) {
                    $filename = time() . '_gallery_' . $index . '_' . Str::slug($validated['name']) . '.' . $image->getClientOriginalExtension();
                    $path = $image->storeAs('products/gallery', $filename, 'public');
                    $galleryPaths[] = $path;
                }
                $validated['gallery_images'] = json_encode($galleryPaths);
            }
            
            // Handle sizes and colors as JSON
            if (isset($validated['sizes'])) {
                $validated['sizes'] = json_encode(array_filter($validated['sizes']));
            }
            if (isset($validated['colors'])) {
                $validated['colors'] = json_encode(array_filter($validated['colors']));
            }
            
            // Generate SKU if not provided
            if (empty($validated['sku'])) {
                $validated['sku'] = 'PRD-' . strtoupper(Str::random(8));
            }
            
            // Create the product
            $product = Product::create($validated);
            
            DB::commit();
            
            return redirect()->route('admin.products.index')
                ->with('success', 'Product created successfully!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            // Delete uploaded images if product creation fails
            if (isset($validated['featured_image'])) {
                Storage::disk('public')->delete($validated['featured_image']);
            }
            if (!empty($galleryPaths)) {
                foreach ($galleryPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
            }
            
            return back()->withInput()
                ->with('error', 'Failed to create product: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified product.
     */
    public function show(Product $product)
    {
        $product->load('category');
        return view('admin.products.show', compact('product'));
    }

    /**
     * Show the form for editing the specified product.
     */
    public function edit(Product $product)
    {
        $categories = Category::where('is_active', true)
            ->orderBy('name')
            ->get();
        
        return view('admin.products.edit', compact('product', 'categories'));
    }

    /**
     * Update the specified product in storage.
     */
    public function update(Request $request, Product $product)
    {
        // Validate the request
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'short_description' => 'nullable|string|max:500',
            'category_id' => 'required|exists:categories,id',
            'sku' => 'nullable|string|unique:products,sku,' . $product->id,
            'price' => 'required|numeric|min:0',
            'compare_price' => 'nullable|numeric|min:0',
            'cost_price' => 'nullable|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'hover_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'gallery_images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'sizes' => 'nullable|array',
            'colors' => 'nullable|array',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_keywords' => 'nullable|string',
        ]);

        DB::beginTransaction();
        
        try {
            // Update slug if name changed
            if ($product->name !== $validated['name']) {
                $validated['slug'] = Str::slug($validated['name']);
                
                // Make slug unique if needed
                $count = Product::where('slug', 'like', $validated['slug'] . '%')
                    ->where('id', '!=', $product->id)
                    ->count();
                if ($count > 0) {
                    $validated['slug'] = $validated['slug'] . '-' . ($count + 1);
                }
            }
            
            // Handle checkboxes
            $validated['is_featured'] = $request->has('is_featured');
            $validated['is_active'] = $request->has('is_active');
            
            // Handle featured image upload
            if ($request->hasFile('featured_image')) {
                // Delete old image
                if ($product->featured_image) {
                    Storage::disk('public')->delete($product->featured_image);
                }

                $featuredImage = $request->file('featured_image');
                $filename = time() . '_' . Str::slug($validated['name']) . '.' . $featuredImage->getClientOriginalExtension();
                $path = $featuredImage->storeAs('products', $filename, 'public');
                $validated['featured_image'] = $path;
            }

            // Handle hover image upload
            if ($request->hasFile('hover_image')) {
                // Delete old hover image
                if ($product->hover_image) {
                    Storage::disk('public')->delete($product->hover_image);
                }

                $hoverImage = $request->file('hover_image');
                $filename = time() . '_hover_' . Str::slug($validated['name']) . '.' . $hoverImage->getClientOriginalExtension();
                $path = $hoverImage->storeAs('products', $filename, 'public');
                $validated['hover_image'] = $path;
            }

            // Handle gallery images
            if ($request->hasFile('gallery_images')) {
                // Delete old gallery images if replacing
                if ($product->gallery_images) {
                    $oldGallery = json_decode($product->gallery_images, true);
                    if (is_array($oldGallery)) {
                        foreach ($oldGallery as $oldPath) {
                            Storage::disk('public')->delete($oldPath);
                        }
                    }
                }
                
                $galleryPaths = [];
                foreach ($request->file('gallery_images') as $index => $image) {
                    $filename = time() . '_gallery_' . $index . '_' . Str::slug($validated['name']) . '.' . $image->getClientOriginalExtension();
                    $path = $image->storeAs('products/gallery', $filename, 'public');
                    $galleryPaths[] = $path;
                }
                $validated['gallery_images'] = json_encode($galleryPaths);
            }
            
            // Handle sizes and colors as JSON
            if (isset($validated['sizes'])) {
                $validated['sizes'] = json_encode(array_filter($validated['sizes']));
            }
            if (isset($validated['colors'])) {
                $validated['colors'] = json_encode(array_filter($validated['colors']));
            }
            
            // Update the product
            $product->update($validated);
            
            DB::commit();
            
            return redirect()->route('admin.products.index')
                ->with('success', 'Product updated successfully!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            
            return back()->withInput()
                ->with('error', 'Failed to update product: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified product from storage.
     */
    public function destroy(Product $product)
    {
        try {
            // Delete images
            if ($product->featured_image) {
                Storage::disk('public')->delete($product->featured_image);
            }

            if ($product->hover_image) {
                Storage::disk('public')->delete($product->hover_image);
            }

            if ($product->gallery_images) {
                $gallery = json_decode($product->gallery_images, true);
                if (is_array($gallery)) {
                    foreach ($gallery as $path) {
                        Storage::disk('public')->delete($path);
                    }
                }
            }
            
            $product->delete();
            
            return redirect()->route('admin.products.index')
                ->with('success', 'Product deleted successfully!');
                
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete product: ' . $e->getMessage());
        }
    }

    /**
     * Toggle product featured status.
     */
    public function toggleFeatured(Product $product)
    {
        $product->update(['is_featured' => !$product->is_featured]);
        
        return back()->with('success', 'Product featured status updated!');
    }

    /**
     * Duplicate a product.
     */
    public function duplicate(Product $product)
    {
        try {
            $newProduct = $product->replicate();
            $newProduct->name = $product->name . ' (Copy)';
            $newProduct->slug = Str::slug($newProduct->name) . '-' . time();
            $newProduct->sku = 'PRD-' . strtoupper(Str::random(8));
            $newProduct->save();
            
            return redirect()->route('admin.products.edit', $newProduct)
                ->with('success', 'Product duplicated successfully!');
                
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to duplicate product: ' . $e->getMessage());
        }
    }
}