<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Helpers\ImageHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class TrendsController extends Controller
{
    /**
     * Display trends dashboard
     */
    public function index()
    {
        $weeklyTrends = $this->getWeeklyTrendsData();
        $seasonalForecasts = $this->getSeasonalForecastsData();
        $fashionNews = $this->getFashionNewsData();
        $predictions = $this->getPredictionsData();
        
        $stats = [
            'weekly_trends' => count($weeklyTrends),
            'seasonal_forecasts' => count($seasonalForecasts),
            'fashion_news' => count($fashionNews),
            'predictions' => count($predictions)
        ];
        
        return view('admin.trends.index', compact('weeklyTrends', 'seasonalForecasts', 'fashionNews', 'predictions', 'stats'));
    }

    /**
     * Weekly trends management
     */
    public function weekly()
    {
        $weeklyTrends = $this->getWeeklyTrendsData();
        return view('admin.trends.weekly', compact('weeklyTrends'));
    }

    /**
     * Create new weekly trend
     */
    public function createWeekly()
    {
        return view('admin.trends.create-weekly');
    }

    /**
     * Store new weekly trend
     */
    public function storeWeekly(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'trend_level' => 'required|in:Rising Fast,Mainstream,Trending',
            'popularity' => 'required|integer|min:0|max:100',
            'key_pieces' => 'required|array|min:1'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'trends');
        }

        // In a real application, you would save this to database
        // For now, we'll simulate success
        return redirect()->route('admin.trends.weekly')
                        ->with('success', 'Weekly trend created successfully!');
    }

    /**
     * Edit weekly trend
     */
    public function editWeekly($id)
    {
        $weeklyTrends = $this->getWeeklyTrendsData();
        $trend = collect($weeklyTrends)->first(function ($item, $key) use ($id) {
            return $key == $id;
        });

        if (!$trend) {
            abort(404);
        }

        return view('admin.trends.edit-weekly', compact('trend', 'id'));
    }

    /**
     * Update weekly trend
     */
    public function updateWeekly(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'trend_level' => 'required|in:Rising Fast,Mainstream,Trending',
            'popularity' => 'required|integer|min:0|max:100',
            'key_pieces' => 'required|array|min:1'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'trends');
        }

        return redirect()->route('admin.trends.weekly')
                        ->with('success', 'Weekly trend updated successfully!');
    }

    /**
     * Delete weekly trend
     */
    public function destroyWeekly($id)
    {
        return redirect()->route('admin.trends.weekly')
                        ->with('success', 'Weekly trend deleted successfully!');
    }

    /**
     * Seasonal forecasts management
     */
    public function seasonal()
    {
        $seasonalForecasts = $this->getSeasonalForecastsData();
        return view('admin.trends.seasonal', compact('seasonalForecasts'));
    }

    /**
     * Create seasonal forecast
     */
    public function createSeasonal()
    {
        return view('admin.trends.create-seasonal');
    }

    /**
     * Store seasonal forecast
     */
    public function storeSeasonal(Request $request)
    {
        $request->validate([
            'season' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'key_trends' => 'required|array|min:1',
            'color_palette' => 'required|array|min:1',
            'must_have_pieces' => 'required|array|min:1'
        ]);

        return redirect()->route('admin.trends.seasonal')
                        ->with('success', 'Seasonal forecast created successfully!');
    }

    /**
     * Fashion news management
     */
    public function news()
    {
        $fashionNews = $this->getFashionNewsData();
        return view('admin.trends.news', compact('fashionNews'));
    }

    /**
     * Create news article
     */
    public function createNews()
    {
        return view('admin.trends.create-news');
    }

    /**
     * Store news article
     */
    public function storeNews(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'excerpt' => 'required|string|max:500',
            'content' => 'required|string|min:500',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'category' => 'required|in:Brand News,Industry News,Global Trends',
            'author' => 'required|string|max:255',
            'read_time' => 'required|integer|min:1|max:30',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'required|string|max:300',
            'keywords' => 'required|string'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'news');
        }

        return redirect()->route('admin.trends.news')
                        ->with('success', 'News article created successfully!');
    }

    /**
     * Edit news article
     */
    public function editNews($slug)
    {
        $fashionNews = $this->getFashionNewsData();
        $article = collect($fashionNews)->firstWhere('slug', $slug);

        if (!$article) {
            abort(404);
        }

        return view('admin.trends.edit-news', compact('article'));
    }

    /**
     * Update news article
     */
    public function updateNews(Request $request, $slug)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'excerpt' => 'required|string|max:500',
            'content' => 'required|string|min:500',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'category' => 'required|in:Brand News,Industry News,Global Trends',
            'author' => 'required|string|max:255',
            'read_time' => 'required|integer|min:1|max:30',
            'meta_title' => 'required|string|max:255',
            'meta_description' => 'required|string|max:300',
            'keywords' => 'required|string'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'news');
        }

        return redirect()->route('admin.trends.news')
                        ->with('success', 'News article updated successfully!');
    }

    /**
     * Delete news article
     */
    public function destroyNews($slug)
    {
        return redirect()->route('admin.trends.news')
                        ->with('success', 'News article deleted successfully!');
    }

    /**
     * Style predictions management
     */
    public function predictions()
    {
        $stylePredictions = $this->getPredictionsData();
        return view('admin.trends.predictions', compact('stylePredictions'));
    }

    /**
     * Create prediction
     */
    public function createPrediction()
    {
        return view('admin.trends.create-prediction');
    }

    /**
     * Store prediction
     */
    public function storePrediction(Request $request)
    {
        $request->validate([
            'timeline' => 'required|string|max:255',
            'prediction' => 'required|string|max:255',
            'description' => 'required|string',
            'confidence' => 'required|integer|min:0|max:100',
            'impact_areas' => 'required|array|min:1'
        ]);

        return redirect()->route('admin.trends.predictions')
                        ->with('success', 'Prediction created successfully!');
    }

    /**
     * Edit seasonal forecast
     */
    public function editSeasonal($slug)
    {
        $seasonalForecasts = $this->getSeasonalForecastsData();
        $forecast = collect($seasonalForecasts)->firstWhere('slug', $slug);

        if (!$forecast) {
            abort(404);
        }

        return view('admin.trends.edit-seasonal', compact('forecast'));
    }

    /**
     * Update seasonal forecast
     */
    public function updateSeasonal(Request $request, $slug)
    {
        $request->validate([
            'season' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'status' => 'required|in:active,draft'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'seasonal');
        }

        return redirect()->route('admin.trends.seasonal')
                        ->with('success', 'Seasonal forecast updated successfully!');
    }

    /**
     * Delete seasonal forecast
     */
    public function destroySeasonal($slug)
    {
        return redirect()->route('admin.trends.seasonal')
                        ->with('success', 'Seasonal forecast deleted successfully!');
    }

    /**
     * Create predictions (renamed for consistency)
     */
    public function createPredictions()
    {
        return view('admin.trends.create-predictions');
    }

    /**
     * Store predictions (renamed for consistency)
     */
    public function storePredictions(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'category' => 'required|in:Technology,Sustainability,Materials,Design Trends',
            'year' => 'required|integer|min:2024|max:2030',
            'confidence' => 'required|integer|min:0|max:100',
            'impact_level' => 'required|in:Revolutionary,Significant,Moderate',
            'timeline' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'predictions');
        }

        return redirect()->route('admin.trends.predictions')
                        ->with('success', 'Style prediction created successfully!');
    }

    /**
     * Edit predictions
     */
    public function editPredictions($slug)
    {
        $stylePredictions = $this->getPredictionsData();
        $prediction = collect($stylePredictions)->firstWhere('slug', $slug);

        if (!$prediction) {
            abort(404);
        }

        return view('admin.trends.edit-predictions', compact('prediction'));
    }

    /**
     * Update predictions
     */
    public function updatePredictions(Request $request, $slug)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'category' => 'required|in:Technology,Sustainability,Materials,Design Trends',
            'year' => 'required|integer|min:2024|max:2030',
            'confidence' => 'required|integer|min:0|max:100',
            'impact_level' => 'required|in:Revolutionary,Significant,Moderate',
            'timeline' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        // Handle image upload using ImageHelper
        if ($request->hasFile('image')) {
            $imagePath = ImageHelper::uploadImage($request->file('image'), 'predictions');
        }

        return redirect()->route('admin.trends.predictions')
                        ->with('success', 'Style prediction updated successfully!');
    }

    /**
     * Delete predictions
     */
    public function destroyPredictions($slug)
    {
        return redirect()->route('admin.trends.predictions')
                        ->with('success', 'Style prediction deleted successfully!');
    }

    /**
     * Get weekly trends data
     */
    private function getWeeklyTrendsData()
    {
        return [
            [
                'slug' => 'rise-cultural-streetwear-australia',
                'title' => 'The Rise of Cultural Streetwear in Australia',
                'description' => 'How heritage-inspired fashion is reshaping Australian streetwear culture',
                'image' => 'trends/cultural-streetwear-rise.jpg',
                'trend_level' => 'Rising Fast',
                'popularity' => 94,
                'key_pieces' => ['Heritage Hoodies', 'Cultural Print Tees', 'Traditional Color Palettes'],
                'created_at' => '2024-08-18 10:15:00',
                'updated_at' => '2024-08-20 16:30:00'
            ],
            [
                'slug' => 'sustainable-fashion-center-stage',
                'title' => 'Sustainable Fashion Takes Center Stage',
                'description' => 'Eco-conscious consumers driving demand for sustainable streetwear options',
                'image' => 'trends/sustainable-center-stage.jpg',
                'trend_level' => 'Mainstream',
                'popularity' => 89,
                'key_pieces' => ['Organic Cotton Basics', 'Recycled Polyester Outerwear', 'Hemp Accessories'],
                'created_at' => '2024-08-15 14:20:00',
                'updated_at' => '2024-08-19 11:45:00'
            ],
            [
                'slug' => 'oversized-silhouettes-tailored-details',
                'title' => 'Oversized Silhouettes with Tailored Details',
                'description' => 'The perfect balance between comfort and structure in modern streetwear',
                'image' => 'trends/oversized-tailored.jpg',
                'trend_level' => 'Trending',
                'popularity' => 87,
                'key_pieces' => ['Structured Hoodies', 'Tailored Joggers', 'Cropped Oversized Tees'],
                'created_at' => '2024-08-12 09:30:00',
                'updated_at' => '2024-08-17 15:15:00'
            ]
        ];
    }

    /**
     * Get seasonal forecasts data
     */
    private function getSeasonalForecastsData()
    {
        return [
            'spring_2024' => [
                'slug' => 'spring-2024-fresh-beginnings',
                'season' => 'Spring',
                'year' => '2024',
                'title' => 'Fresh Beginnings: Spring Streetwear Trends',
                'description' => 'As Australia transitions into spring, streetwear embraces lighter fabrics, vibrant colors, and fresh cultural inspirations.',
                'image' => 'seasonal/spring-2024-forecast.jpg',
                'status' => 'active',
                'key_trends' => [
                    'Lightweight layering pieces',
                    'Tropical color palettes',
                    'Breathable cultural graphics',
                    'Sun-protection streetwear'
                ],
                'color_palette' => ['Sage Green', 'Ocean Blue', 'Sunset Orange', 'Natural Beige'],
                'must_have_pieces' => [
                    'Lightweight zip hoodies',
                    'UV-protection tees',
                    'Linen-blend joggers',
                    'Breathable caps'
                ],
                'created_at' => '2024-08-15 10:00:00',
                'updated_at' => '2024-08-20 15:30:00'
            ],
            'summer_2024' => [
                'slug' => 'summer-2024-heat-wave-style',
                'season' => 'Summer',
                'year' => '2024',
                'title' => 'Heat Wave Style: Summer Streetwear Essentials',
                'description' => 'Australian summer demands smart fabric choices and bold style statements that beat the heat.',
                'image' => 'seasonal/summer-2024-forecast.jpg',
                'status' => 'active',
                'key_trends' => [
                    'Moisture-wicking materials',
                    'Bold tropical prints',
                    'Minimal layering',
                    'Statement accessories'
                ],
                'color_palette' => ['Electric Blue', 'Coral Pink', 'Lime Green', 'Pure White'],
                'must_have_pieces' => [
                    'Performance tees',
                    'Board shorts',
                    'Lightweight bombers',
                    'UV sunglasses'
                ],
                'created_at' => '2024-07-10 14:00:00',
                'updated_at' => '2024-08-18 09:45:00'
            ],
            'autumn_2024' => [
                'slug' => 'autumn-2024-transitional-layers',
                'season' => 'Autumn',
                'year' => '2024',
                'title' => 'Transitional Layers: Autumn Streetwear Evolution',
                'description' => 'Australian autumn brings the perfect opportunity for layering and transitional pieces that bridge seasons.',
                'image' => 'seasonal/autumn-2024-forecast.jpg',
                'status' => 'active',
                'key_trends' => [
                    'Transitional layering systems',
                    'Earth tone color palettes',
                    'Technical fabric integration',
                    'Heritage-inspired details'
                ],
                'color_palette' => ['Burnt Orange', 'Deep Forest', 'Rust Red', 'Warm Taupe'],
                'must_have_pieces' => [
                    'Layerable zip hoodies',
                    'Technical fleece joggers',
                    'Weather-resistant windbreakers',
                    'Wool-blend beanies'
                ],
                'created_at' => '2024-06-20 11:30:00',
                'updated_at' => '2024-08-15 16:20:00'
            ],
            'winter_2024' => [
                'slug' => 'winter-2024-cozy-comfort',
                'season' => 'Winter',
                'year' => '2024',
                'title' => 'Cozy Comfort: Winter Streetwear Mastery',
                'description' => 'Australian winter streetwear combines warmth with style, featuring heavy layering and premium comfort materials.',
                'image' => 'seasonal/winter-2024-forecast.jpg',
                'status' => 'draft',
                'key_trends' => [
                    'Premium fleece construction',
                    'Oversized comfort silhouettes',
                    'Dark color dominance',
                    'Functional pocket systems'
                ],
                'color_palette' => ['Charcoal Black', 'Deep Navy', 'Steel Grey', 'Burgundy'],
                'must_have_pieces' => [
                    'Heavy-weight hoodies',
                    'Insulated track pants',
                    'Puffer vest layers',
                    'Thermal base layers'
                ],
                'created_at' => '2024-05-15 08:45:00',
                'updated_at' => '2024-08-10 13:15:00'
            ]
        ];
    }

    /**
     * Get fashion news data
     */
    private function getFashionNewsData()
    {
        return [
            [
                'title' => 'Edition By Punjab Featured in Australian Fashion Week',
                'slug' => 'edition-by-punjab-featured-australian-fashion-week',
                'excerpt' => 'Our cultural fusion collection receives acclaim at Melbourne Fashion Week for innovative heritage-meets-streetwear designs.',
                'content' => 'Edition By Punjab made history at Melbourne Fashion Week 2024, showcasing a groundbreaking collection that seamlessly blends traditional Punjabi heritage with contemporary Australian streetwear aesthetics...',
                'image' => 'news/fashion-week-feature.jpg',
                'published_date' => '2024-09-01',
                'category' => 'Brand News',
                'read_time' => 5,
                'author' => 'Sarah Mitchell',
                'meta_title' => 'Edition By Punjab Shines at Australian Fashion Week 2024 | Melbourne Fashion',
                'meta_description' => 'Edition By Punjab\'s cultural fusion streetwear collection receives critical acclaim at Melbourne Fashion Week 2024.',
                'keywords' => 'Australian Fashion Week, Melbourne Fashion Week, streetwear Australia',
                'created_at' => '2024-09-01 08:30:00',
                'updated_at' => '2024-09-01 14:15:00'
            ],
            [
                'title' => 'Melbourne Streetwear Scene Explodes: Local Brands Rise to Prominence',
                'slug' => 'melbourne-streetwear-scene-explodes-local-brands-prominence',
                'excerpt' => 'Melbourne\'s streetwear culture reaches new heights as local brands gain national recognition and celebrity endorsements.',
                'content' => 'Melbourne\'s streetwear landscape has undergone a remarkable transformation over the past year, with local brands experiencing unprecedented growth and recognition both domestically and internationally...',
                'image' => 'news/melbourne-streetwear-explosion.jpg',
                'published_date' => '2024-08-20',
                'category' => 'Industry News',
                'read_time' => 6,
                'author' => 'Jessica Wong',
                'meta_title' => 'Melbourne Streetwear Brands 2024 | Australian Street Fashion Trends',
                'meta_description' => 'Melbourne\'s streetwear scene explodes with local brands gaining national recognition.',
                'keywords' => 'Melbourne streetwear, Australian street fashion, Melbourne fashion brands',
                'created_at' => '2024-08-20 12:00:00',
                'updated_at' => '2024-08-20 16:45:00'
            ],
            // Add more news items as needed
        ];
    }

    /**
     * Get predictions data
     */
    private function getPredictionsData()
    {
        return [
            [
                'slug' => 'cultural-heritage-graphics-boom-2024',
                'title' => 'Cultural Heritage Graphics Boom',
                'description' => 'Expect to see more brands incorporating traditional cultural elements into contemporary streetwear designs.',
                'image' => 'predictions/cultural-heritage-graphics.jpg',
                'category' => 'Design Trends',
                'year' => 2024,
                'confidence' => 92,
                'impact_level' => 'Significant',
                'timeline' => 'Next 3 Months',
                'impact_areas' => ['Print Design', 'Color Palettes', 'Typography'],
                'created_at' => '2024-08-10 09:00:00',
                'updated_at' => '2024-08-20 14:30:00'
            ],
            [
                'slug' => 'sustainable-materials-mainstream-2024',
                'title' => 'Sustainable Materials Mainstream',
                'description' => 'Organic cotton, recycled polyester, and hemp will become standard materials rather than premium options.',
                'image' => 'predictions/sustainable-materials.jpg',
                'category' => 'Sustainability',
                'year' => 2024,
                'confidence' => 88,
                'impact_level' => 'Revolutionary',
                'timeline' => 'Next 6 Months',
                'impact_areas' => ['Material Innovation', 'Pricing', 'Consumer Education'],
                'created_at' => '2024-07-25 11:15:00',
                'updated_at' => '2024-08-18 16:45:00'
            ],
            [
                'slug' => 'customization-personalization-trend-2025',
                'title' => 'Customization and Personalization Revolution',
                'description' => 'Brands will offer more customization options, allowing customers to personalize cultural and style elements.',
                'image' => 'predictions/customization-revolution.jpg',
                'category' => 'Technology',
                'year' => 2025,
                'confidence' => 85,
                'impact_level' => 'Significant',
                'timeline' => 'Next 12 Months',
                'impact_areas' => ['Production Methods', 'Customer Experience', 'Brand Differentiation'],
                'created_at' => '2024-06-30 13:20:00',
                'updated_at' => '2024-08-15 10:00:00'
            ],
            [
                'slug' => 'smart-fabric-integration-2025',
                'title' => 'Smart Fabric Integration',
                'description' => 'Integration of smart textiles and wearable technology into everyday streetwear pieces for enhanced functionality.',
                'image' => 'predictions/smart-fabric-tech.jpg',
                'category' => 'Technology',
                'year' => 2025,
                'confidence' => 78,
                'impact_level' => 'Revolutionary',
                'timeline' => 'Next 18 Months',
                'impact_areas' => ['Material Innovation', 'Health Monitoring', 'User Experience'],
                'created_at' => '2024-05-20 15:45:00',
                'updated_at' => '2024-08-12 12:30:00'
            ],
            [
                'slug' => 'circular-fashion-economy-2025',
                'title' => 'Circular Fashion Economy',
                'description' => 'Complete shift towards circular fashion models with take-back programs and material recycling becoming industry standard.',
                'image' => 'predictions/circular-fashion-economy.jpg',
                'category' => 'Sustainability',
                'year' => 2025,
                'confidence' => 82,
                'impact_level' => 'Revolutionary',
                'timeline' => 'Next 24 Months',
                'impact_areas' => ['Business Models', 'Environmental Impact', 'Consumer Behavior'],
                'created_at' => '2024-04-15 08:30:00',
                'updated_at' => '2024-08-08 17:20:00'
            ]
        ];
    }
}