<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use App\Models\Category;
use App\Models\CompanyInfo;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    /**
     * Display blog listing page
     */
    public function index(Request $request)
    {
        $query = Blog::published()->latest();
        
        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function($q) use ($searchTerm) {
                $q->where('title', 'LIKE', '%' . $searchTerm . '%')
                  ->orWhere('content', 'LIKE', '%' . $searchTerm . '%')
                  ->orWhere('excerpt', 'LIKE', '%' . $searchTerm . '%');
            });
        }
        
        // Tag filtering
        if ($request->has('tag') && !empty($request->tag)) {
            $query->whereJsonContains('tags', $request->tag);
        }
        
        $blogs = $query->paginate(12);
        $featuredBlogs = Blog::published()->featured()->latest()->take(3)->get();
        $recentBlogs = Blog::published()->latest()->take(5)->get();
        $popularTags = $this->getPopularTags();
        
        $categories = Category::active()->get();
        $companyInfo = CompanyInfo::first();
        
        return view('blogs.index', compact(
            'blogs',
            'featuredBlogs', 
            'recentBlogs',
            'popularTags',
            'categories',
            'companyInfo'
        ));
    }
    
    /**
     * Display individual blog post
     */
    public function show(Blog $blog)
    {
        // Only show published blogs
        if (!$blog->is_published) {
            abort(404);
        }
        
        // Increment view count
        $blog->incrementViews();
        
        // Get related blogs
        $relatedBlogs = Blog::published()
            ->where('id', '!=', $blog->id)
            ->latest()
            ->take(3)
            ->get();
            
        $recentBlogs = Blog::published()
            ->where('id', '!=', $blog->id)
            ->latest()
            ->take(5)
            ->get();
            
        $popularTags = $this->getPopularTags();
        
        $categories = Category::active()->get();
        $companyInfo = CompanyInfo::first();
        
        return view('blogs.show', compact(
            'blog',
            'relatedBlogs',
            'recentBlogs', 
            'popularTags',
            'categories',
            'companyInfo'
        ));
    }
    
    /**
     * Get popular tags
     */
    private function getPopularTags()
    {
        $blogs = Blog::published()
            ->whereNotNull('tags')
            ->get();
            
        $allTags = [];
        foreach ($blogs as $blog) {
            if ($blog->tags) {
                $allTags = array_merge($allTags, $blog->tags);
            }
        }
        
        $tagCounts = array_count_values($allTags);
        arsort($tagCounts);
        
        return array_slice($tagCounts, 0, 15, true);
    }
}
