<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\Product;
use App\Models\Coupon;
use App\Models\UserDiscount;
use App\Models\ShippingZone;
use App\Models\TaxRate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    /**
     * Return the identifier array used to find cart rows for the current visitor/user.
     * Either ['user_id' => id] for logged-in users or ['session_id' => '...'] for guests.
     */
    private function getCartIdentifier()
    {
        if (Auth::check()) {
            return ['user_id' => Auth::id()];
        }

        // Create or get session ID for guest users
        if (!Session::has('cart_session')) {
            Session::put('cart_session', uniqid('cart_', true));
        }

        return ['session_id' => Session::get('cart_session')];
    }

    /**
     * Display cart page with dynamic tax & shipping calculation.
     */
    public function index(Request $request)
    {
        $cartItems = Cart::with('product')
            ->where($this->getCartIdentifier())
            ->get();

        $subtotal = 0;
        foreach ($cartItems as $item) {
            $item->total = $item->price * $item->quantity;
            $subtotal += $item->total;
        }

        // Get shipping info from session (or sensible defaults)
        $shippingCity = Session::get('shipping_city', 'Sydney');
        $shippingState = Session::get('shipping_state', 'NSW');
        $shippingMethod = Session::get('shipping_method', 'standard');

        // Dynamic tax (using TaxRate model if available)
        $taxRate = null;
        $tax = 0.0;
        if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
            $taxRate = TaxRate::findByCity($shippingCity, $shippingState);
        }
        if ($taxRate && method_exists($taxRate, 'calculateTax')) {
            $tax = $taxRate->calculateTax($subtotal);
        } else {
            // Fallback: flat 10%
            $tax = $subtotal * 0.10;
        }

        // Dynamic shipping (using ShippingZone model if available)
        $shippingZone = null;
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
            $shippingZone = ShippingZone::findByCity($shippingCity, $shippingState);
        }
        if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
            $shippingZone = ShippingZone::getDefault();
        }
        if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
            $shipping = $shippingZone->getShippingRate($shippingMethod, $subtotal);
        } else {
            // Fallback flat rate
            $shipping = $this->calculateShipping($subtotal, $shippingCity, $shippingState, $shippingMethod);
        }

        // Discounts
        $discount = $this->calculateDiscountAmount($subtotal);
        $couponSession = Session::get('coupon');
        $discountSession = Session::get('discount');

        $total = max(0, $subtotal + $tax + $shipping - $discount);

        // Get all active shipping zones for dropdown - ALWAYS return a Collection
        $shippingZones = collect([]); // Initialize as empty collection
        
        try {
            if (class_exists(ShippingZone::class)) {
                if (method_exists(ShippingZone::class, 'active')) {
                    $shippingZones = ShippingZone::active()
                        ->orderBy('state')
                        ->orderBy('city')
                        ->get();
                } else {
                    // If active() method doesn't exist, try getting all active zones directly
                    $shippingZones = ShippingZone::where('is_active', true)
                        ->orderBy('state')
                        ->orderBy('city')
                        ->get();
                }
            }
        } catch (\Exception $e) {
            // Log error but continue with empty collection
            \Log::error('Error loading shipping zones: ' . $e->getMessage());
            $shippingZones = collect([]);
        }
        
        // Get all active tax rates for reference
        $taxRates = collect([]);
        try {
            if (class_exists(TaxRate::class)) {
                $taxRates = TaxRate::active()
                    ->orderBy('state')
                    ->orderBy('city')
                    ->get();
            }
        } catch (\Exception $e) {
            \Log::error('Error loading tax rates: ' . $e->getMessage());
            $taxRates = collect([]);
        }
        
        // Create combined location data with both tax and shipping info
        $locationData = $this->getCombinedLocationData($shippingZones, $taxRates);

        return view('shop.cart', compact(
            'cartItems', 'subtotal', 'tax', 'shipping', 'discount', 'total',
            'couponSession', 'discountSession', 'shippingZones', 'shippingCity',
            'shippingState', 'shippingMethod', 'shippingZone', 'taxRate', 'taxRates',
            'locationData'
        ));
    }

    /**
     * Update shipping selection (city/state/method) — stores values in session and returns recalculated totals (JSON).
     */
    public function updateShipping(Request $request)
    {
        $validated = $request->validate([
            'city' => 'required|string',
            'state' => 'required|string',
            'method' => 'required|in:standard,express,priority'
        ]);

        Session::put('shipping_city', $validated['city']);
        Session::put('shipping_state', $validated['state']);
        Session::put('shipping_method', $validated['method']);

        // Recalculate subtotal
        $cartItems = Cart::with('product')
            ->where($this->getCartIdentifier())
            ->get();

        $subtotal = 0;
        foreach ($cartItems as $item) {
            $subtotal += $item->price * $item->quantity;
        }

        // Tax
        $taxRate = null;
        $tax = 0.0;
        if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
            $taxRate = TaxRate::findByCity($validated['city'], $validated['state']);
        }
        if ($taxRate && method_exists($taxRate, 'calculateTax')) {
            $tax = $taxRate->calculateTax($subtotal);
        } else {
            $tax = $subtotal * 0.10;
        }

        // Shipping
        $shippingZone = null;
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
            $shippingZone = ShippingZone::findByCity($validated['city'], $validated['state']);
        }
        if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
            $shippingZone = ShippingZone::getDefault();
        }
        if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
            $shipping = $shippingZone->getShippingRate($validated['method'], $subtotal);
        } else {
            $shipping = $this->calculateShipping($subtotal, $validated['city'], $validated['state'], $validated['method']);
        }

        $discount = $this->calculateDiscountAmount($subtotal);
        $total = max(0, $subtotal + $tax + $shipping - $discount);

        return response()->json([
            'success' => true,
            'shipping' => number_format($shipping, 2),
            'tax' => number_format($tax, 2),
            'total' => number_format($total, 2),
            'shipping_raw' => $shipping,
            'tax_raw' => $tax,
            'total_raw' => $total,
            'tax_rate_name' => $taxRate ? $taxRate->tax_name : 'GST',
            'tax_rate_formatted' => $taxRate ? $taxRate->formatted_rate : '10%',
            'tax_rate_percentage' => $taxRate ? $taxRate->rate : 10,
            'shipping_zone' => $shippingZone ? [
                'city' => $shippingZone->city,
                'state' => $shippingZone->state,
                'free_threshold' => $shippingZone->free_shipping_threshold
            ] : null
        ]);
    }

    /**
     * Add product to cart (handles attributes as array).
     */
    public function add(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
            'size' => 'nullable|string',
            'color' => 'nullable|string'
        ]);

        $product = Product::findOrFail($request->product_id);

        // Check stock
        if ($product->stock < $request->quantity) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient stock available'
            ], 400);
        }

        // Prepare attributes as array (not JSON string)
        $attributes = [];
        if ($request->size) {
            $attributes['size'] = $request->size;
        }
        if ($request->color) {
            $attributes['color'] = $request->color;
        }

        // Find existing cart items for this product for current cart identifier
        $cartQuery = Cart::where($this->getCartIdentifier())
            ->where('product_id', $product->id);

        $existingItems = $cartQuery->get();
        $cartItem = null;

        foreach ($existingItems as $item) {
            // Direct array comparison (Cart model should cast attributes to array)
            if ($item->attributes == $attributes) {
                $cartItem = $item;
                break;
            }
        }

        if ($cartItem) {
            // Update quantity
            $newQuantity = $cartItem->quantity + $request->quantity;

            if ($product->stock < $newQuantity) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot add more items. Stock limit reached.'
                ], 400);
            }

            $cartItem->quantity = $newQuantity;
            $cartItem->save();
        } else {
            // Add new item - pass attributes as array (ensure Cart model has $casts = ['attributes' => 'array'])
            $cartData = array_merge($this->getCartIdentifier(), [
                'product_id' => $product->id,
                'quantity' => $request->quantity,
                'price' => $product->price,
                'attributes' => !empty($attributes) ? $attributes : null
            ]);

            Cart::create($cartData);
        }

        // Get updated cart count
        $cartCount = Cart::where($this->getCartIdentifier())->sum('quantity');

        return response()->json([
            'success' => true,
            'message' => 'Product added to cart',
            'cartCount' => $cartCount
        ]);
    }

    /**
     * Update cart item quantity and return recalculated totals.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1'
        ]);

        $cartItem = Cart::where($this->getCartIdentifier())
            ->where('id', $id)
            ->firstOrFail();

        // Check stock
        if ($cartItem->product && $cartItem->product->stock < $request->quantity) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient stock available'
            ], 400);
        }

        $cartItem->quantity = $request->quantity;
        $cartItem->save();

        // Recalculate subtotal
        $cartItems = Cart::with('product')
            ->where($this->getCartIdentifier())
            ->get();

        $subtotal = 0;
        foreach ($cartItems as $item) {
            $subtotal += $item->price * $item->quantity;
        }

        // Get shipping info from session
        $shippingCity = Session::get('shipping_city', 'Sydney');
        $shippingState = Session::get('shipping_state', 'NSW');
        $shippingMethod = Session::get('shipping_method', 'standard');

        // Tax
        $taxRate = null;
        $tax = 0.0;
        if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
            $taxRate = TaxRate::findByCity($shippingCity, $shippingState);
        }
        if ($taxRate && method_exists($taxRate, 'calculateTax')) {
            $tax = $taxRate->calculateTax($subtotal);
        } else {
            $tax = $subtotal * 0.10;
        }

        // Shipping
        $shippingZone = null;
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
            $shippingZone = ShippingZone::findByCity($shippingCity, $shippingState);
        }
        if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
            $shippingZone = ShippingZone::getDefault();
        }
        if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
            $shipping = $shippingZone->getShippingRate($shippingMethod, $subtotal);
        } else {
            $shipping = $this->calculateShipping($subtotal, $shippingCity, $shippingState, $shippingMethod);
        }

        $discount = $this->calculateDiscountAmount($subtotal);
        $total = max(0, $subtotal + $tax + $shipping - $discount);

        return response()->json([
            'success' => true,
            'message' => 'Cart updated',
            'subtotal' => number_format($subtotal, 2),
            'tax' => number_format($tax, 2),
            'shipping' => number_format($shipping, 2),
            'discount' => number_format($discount, 2),
            'total' => number_format($total, 2)
        ]);
    }

    /**
     * Remove item from cart.
     */
    public function remove($id)
    {
        $cartItem = Cart::where($this->getCartIdentifier())
            ->where('id', $id)
            ->firstOrFail();

        $cartItem->delete();

        return redirect()->route('cart.index')
            ->with('success', 'Item removed from cart');
    }

    /**
     * Apply coupon/user-specific discount.
     * Accepts both normal form submissions and AJAX requests (returns JSON).
     */
    public function applyCoupon(Request $request)
    {
        $request->validate([
            'coupon_code' => 'required|string'
        ]);

        // Check for user discount
        $userDiscount = UserDiscount::where('discount_code', $request->coupon_code)
            ->where('email', auth()->user()->email ?? $request->email)
            ->first();

        if ($userDiscount) {
            if (method_exists($userDiscount, 'isValid') && !$userDiscount->isValid()) {
                if ($request->wantsJson() || $request->ajax()) {
                    return response()->json(['success' => false, 'message' => 'This discount code has already been used or has expired.'], 400);
                }
                return back()->with('error', 'This discount code has already been used or has expired.');
            }

            // Apply discount to session
            Session::put('discount', [
                'code' => $userDiscount->discount_code,
                'percentage' => $userDiscount->discount_percentage,
                'type' => $userDiscount->discount_type ?? 'percentage'
            ]);

            if ($request->wantsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Discount applied successfully!',
                    'discount' => $userDiscount->discount_percentage
                ]);
            }

            return back()->with('success', 'Discount applied successfully! You saved ' . $userDiscount->discount_percentage . '%');
        }

        // Check for regular coupon codes
        $coupon = Coupon::where('code', $request->coupon_code)
            ->where('is_active', true)
            ->first();

        if (!$coupon) {
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['success' => false, 'message' => 'Invalid coupon code.'], 400);
            }
            return back()->with('error', 'Invalid coupon code.');
        }

        if (method_exists($coupon, 'isValid') && !$coupon->isValid()) {
            if ($request->wantsJson() || $request->ajax()) {
                return response()->json(['success' => false, 'message' => 'This coupon has expired or reached its usage limit.'], 400);
            }
            return back()->with('error', 'This coupon has expired or reached its usage limit.');
        }

        // Apply coupon to session (store relevant details)
        Session::put('coupon', [
            'code' => $coupon->code,
            'discount' => $coupon->discount_amount ?? 0,
            'type' => $coupon->discount_type ?? 'fixed'
        ]);

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Coupon applied successfully!']);
        }

        return back()->with('success', 'Coupon applied successfully!');
    }

    /**
     * Remove applied coupon/discount.
     */
    public function removeCoupon()
    {
        Session::forget('coupon');
        Session::forget('discount');

        return redirect()->route('cart.index')
            ->with('success', 'Coupon/Discount removed');
    }

    /**
     * Return the total count of items in current cart (for UI badge updates).
     */
    public function getCartCount()
    {
        $count = Cart::where($this->getCartIdentifier())->sum('quantity');

        return response()->json(['count' => $count]);
    }

    /**
     * Merge guest cart into authenticated user's cart after login.
     */
    public function mergeGuestCart()
    {
        if (!Auth::check() || !Session::has('cart_session')) {
            return response()->json(['success' => false]);
        }

        // Get guest cart items
        $guestItems = Cart::where('session_id', Session::get('cart_session'))->get();

        foreach ($guestItems as $item) {
            // Check if item already exists in user's cart (compare attributes properly)
            $existingItems = Cart::where('user_id', Auth::id())
                ->where('product_id', $item->product_id)
                ->get();

            $existingItem = null;
            foreach ($existingItems as $existing) {
                if ($existing->attributes == $item->attributes) {
                    $existingItem = $existing;
                    break;
                }
            }

            if ($existingItem) {
                // Merge quantities
                $existingItem->quantity += $item->quantity;
                $existingItem->save();
                $item->delete();
            } else {
                // Transfer to user's cart
                $item->user_id = Auth::id();
                $item->session_id = null;
                $item->save();
            }
        }

        Session::forget('cart_session');

        return response()->json(['success' => true]);
    }

    /**
     * Checkout page - validates cart, checks stock and calculates totals including discounts.
     */
    public function checkout(Request $request)
    {
        $cartItems = Cart::with('product')
            ->where($this->getCartIdentifier())
            ->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('info', 'Your cart is empty.');
        }

        // Validate stock availability
        foreach ($cartItems as $item) {
            if (!$item->product || $item->product->stock < $item->quantity) {
                return redirect()->route('cart.index')
                    ->with('error', 'One or more items in your cart are out of stock or insufficient quantity.');
            }
        }

        $subtotal = 0;
        foreach ($cartItems as $item) {
            $subtotal += $item->price * $item->quantity;
        }

        // Shipping info from session
        $shippingCity = Session::get('shipping_city', 'Sydney');
        $shippingState = Session::get('shipping_state', 'NSW');
        $shippingMethod = Session::get('shipping_method', 'standard');

        // Tax
        $taxRate = null;
        $tax = 0.0;
        if (class_exists(TaxRate::class) && method_exists(TaxRate::class, 'findByCity')) {
            $taxRate = TaxRate::findByCity($shippingCity, $shippingState);
        }
        if ($taxRate && method_exists($taxRate, 'calculateTax')) {
            $tax = $taxRate->calculateTax($subtotal);
        } else {
            $tax = $subtotal * 0.10;
        }

        // Shipping
        $shippingZone = null;
        if (class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'findByCity')) {
            $shippingZone = ShippingZone::findByCity($shippingCity, $shippingState);
        }
        if (!$shippingZone && class_exists(ShippingZone::class) && method_exists(ShippingZone::class, 'getDefault')) {
            $shippingZone = ShippingZone::getDefault();
        }
        if ($shippingZone && method_exists($shippingZone, 'getShippingRate')) {
            $shipping = $shippingZone->getShippingRate($shippingMethod, $subtotal);
        } else {
            $shipping = $this->calculateShipping($subtotal, $shippingCity, $shippingState, $shippingMethod);
        }

        $discount = $this->calculateDiscountAmount($subtotal);
        $total = max(0, $subtotal + $tax + $shipping - $discount);

        // Pass session coupon/discount info to view so user can see applied codes
        $couponSession = Session::get('coupon');
        $discountSession = Session::get('discount');

        return view('shop.checkout', compact(
            'cartItems', 'subtotal', 'tax', 'shipping', 'discount', 'total', 'couponSession', 'discountSession'
        ));
    }

    /**
     * Default shipping calculation fallback (used if ShippingZone model/methods are unavailable).
     * You can expand logic here (per-city overrides, weight-based, free shipping threshold, etc.).
     *
     * @param float $subtotal
     * @param string|null $city
     * @param string|null $state
     * @param string|null $method
     * @return float
     */
    private function calculateShipping($subtotal, $city = null, $state = null, $method = null)
    {
        // Example: free shipping over $100
        if ($subtotal >= 100) {
            return 0;
        }

        // Example method-based adjustments
        if ($method === 'express') {
            return 20; // express
        } elseif ($method === 'priority') {
            return 30; // priority
        }

        // Default flat rate
        return 10;
    }

    /**
     * Calculate discount amount based on session values.
     * Supports:
     *  - Session 'discount' => ['code' => ..., 'percentage' => X, 'type' => 'percentage'|'fixed']
     *  - Session 'coupon'   => ['code' => ..., 'discount' => amount OR percentage, 'type' => 'fixed'|'percentage']
     */
    private function calculateDiscountAmount($subtotal)
    {
        $discountAmount = 0;

        // User-specific discount has priority (session key: 'discount')
        $userDisc = Session::get('discount');
        if ($userDisc && is_array($userDisc)) {
            $type = $userDisc['type'] ?? 'percentage';
            if ($type === 'fixed') {
                $discountAmount = floatval($userDisc['percentage'] ?? ($userDisc['discount'] ?? 0));
            } else {
                // treat 'percentage' or unspecified as percentage
                $pct = floatval($userDisc['percentage'] ?? 0);
                $discountAmount = ($pct / 100) * $subtotal;
            }

            return round(max(0, $discountAmount), 2);
        }

        // Regular coupon
        $couponSess = Session::get('coupon');
        if ($couponSess && is_array($couponSess)) {
            $type = $couponSess['type'] ?? 'fixed';
            if ($type === 'fixed') {
                $discountAmount = floatval($couponSess['discount'] ?? 0);
            } else {
                // percentage
                $pct = floatval($couponSess['discount'] ?? 0);
                $discountAmount = ($pct / 100) * $subtotal;
            }

            return round(max(0, $discountAmount), 2);
        }

        // Backwards compatibility: older code may have stored coupon as string code
        if (Session::has('coupon') && !is_array(Session::get('coupon'))) {
            $code = Session::get('coupon');
            $coupon = Coupon::where('code', $code)->where('is_active', true)->first();
            if ($coupon && method_exists($coupon, 'isValid') && $coupon->isValid()) {
                // if the Coupon model has calculateDiscount, use that; else compute here
                if (method_exists($coupon, 'calculateDiscount')) {
                    return round($coupon->calculateDiscount($subtotal), 2);
                } else {
                    if (($coupon->discount_type ?? 'fixed') === 'percentage') {
                        return round(($coupon->discount_amount / 100) * $subtotal, 2);
                    }
                    return round($coupon->discount_amount, 2);
                }
            } else {
                // Remove invalid legacy coupon
                Session::forget('coupon');
            }
        }

        return 0;
    }
    
    /**
     * Get combined location data with both tax and shipping information
     */
    private function getCombinedLocationData($shippingZones, $taxRates)
    {
        // Use a plain array instead of Collection to avoid the indirect modification issue
        $cityStateMap = [];
        
        // Add shipping zones
        foreach ($shippingZones as $zone) {
            $key = $zone->city . '_' . $zone->state;
            $cityStateMap[$key] = [
                'city' => $zone->city,
                'state' => $zone->state,
                'shipping' => $zone,
                'tax' => null
            ];
        }
        
        // Add tax rates (merge with existing shipping data if present)
        foreach ($taxRates as $taxRate) {
            $key = $taxRate->city . '_' . $taxRate->state;
            if (isset($cityStateMap[$key])) {
                $cityStateMap[$key]['tax'] = $taxRate;
            } else {
                $cityStateMap[$key] = [
                    'city' => $taxRate->city,
                    'state' => $taxRate->state,
                    'shipping' => null,
                    'tax' => $taxRate
                ];
            }
        }
        
        // Convert to grouped format by state
        $locationsByState = [];
        
        foreach ($cityStateMap as $location) {
            $state = $location['state'];
            
            if (!isset($locationsByState[$state])) {
                $locationsByState[$state] = [];
            }
            
            $locationsByState[$state][] = [
                'city' => $location['city'],
                'state' => $location['state'],
                'tax_rate' => $location['tax'] ? $location['tax']->formatted_rate : null,
                'tax_name' => $location['tax'] ? $location['tax']->tax_name : null,
                'tax_percentage' => $location['tax'] ? $location['tax']->rate : null,
                'free_shipping_threshold' => $location['shipping'] ? $location['shipping']->free_shipping_threshold : null,
                'has_shipping' => $location['shipping'] !== null,
                'has_tax' => $location['tax'] !== null
            ];
        }
        
        // Sort locations within each state and convert to Collection for consistency
        $result = collect([]);
        ksort($locationsByState); // Sort states alphabetically
        
        foreach ($locationsByState as $state => $locations) {
            usort($locations, function($a, $b) {
                return strcmp($a['city'], $b['city']);
            });
            $result->put($state, collect($locations));
        }
        
        return $result;
    }
}