<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\CommunityPost;

class CommunityController extends Controller
{
    /**
     * Community main page with user-generated content
     */
    public function index()
    {
        // Get approved community posts
        $communityPosts = CommunityPost::approved()
            ->orderBy('featured', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit(6)
            ->get()
            ->map(function ($post) {
                return [
                    'id' => $post->id,
                    'user' => $post->name,
                    'location' => $post->location,
                    'image' => $post->image_path,
                    'caption' => $post->caption,
                    'product' => $post->products_featured ?? 'Edition By Punjab Style',
                    'likes' => $post->likes,
                    'instagram_url' => $post->instagram_url ?? '#'
                ];
            });

        // If no real posts exist, show mock data
        if ($communityPosts->isEmpty()) {
            $communityPosts = collect([
                [
                    'id' => 1,
                    'user' => 'Sarah M.',
                    'location' => 'Melbourne',
                    'image' => 'community/post-1.jpg',
                    'caption' => 'Loving my new Edition By Punjab hoodie! Perfect for Melbourne\'s unpredictable weather 🔥 #EditionByPunjab #MelbourneStyle',
                    'product' => 'Cultural Fusion Hoodie',
                    'likes' => 124,
                    'instagram_url' => '#'
                ],
                [
                    'id' => 2,
                    'user' => 'Alex K.',
                    'location' => 'Sydney',
                    'image' => 'community/post-2.jpg',
                    'caption' => 'Beach to brunch vibes with @editionbypunjab 🌊 Quality and comfort in one! #StreetStyle #SydneyFashion',
                    'product' => 'Sustainable Tee',
                    'likes' => 89,
                    'instagram_url' => '#'
                ],
                [
                    'id' => 3,
                    'user' => 'Priya S.',
                    'location' => 'Brisbane',
                    'image' => 'community/post-3.jpg',
                    'caption' => 'Representing my heritage with style! Love brands that celebrate culture 🙏 #CulturalFashion #Representation',
                    'product' => 'Heritage Print Tee',
                    'likes' => 203,
                    'instagram_url' => '#'
                ]
            ]);
        }

        $featuredProducts = Product::where('is_featured', true)->limit(6)->get();

        return view('community.index', compact('communityPosts', 'featuredProducts'));
    }

    /**
     * Style submission page
     */
    public function submit()
    {
        return view('community.submit');
    }

    /**
     * Style gallery page
     */
    public function gallery()
    {
        $galleries = [
            'street-style' => [
                'title' => 'Street Style Spotlight',
                'description' => 'Real customers, real style',
                'images' => [
                    ['src' => 'gallery/street-1.jpg', 'user' => 'Jamie L.', 'location' => 'Melbourne'],
                    ['src' => 'gallery/street-2.jpg', 'user' => 'Chris M.', 'location' => 'Sydney'],
                    ['src' => 'gallery/street-3.jpg', 'user' => 'Aisha P.', 'location' => 'Brisbane'],
                    ['src' => 'gallery/street-4.jpg', 'user' => 'Tom R.', 'location' => 'Perth']
                ]
            ],
            'cultural-pride' => [
                'title' => 'Cultural Pride Collection',
                'description' => 'Celebrating heritage through fashion',
                'images' => [
                    ['src' => 'gallery/cultural-1.jpg', 'user' => 'Simran K.', 'location' => 'Adelaide'],
                    ['src' => 'gallery/cultural-2.jpg', 'user' => 'Raj S.', 'location' => 'Melbourne'],
                    ['src' => 'gallery/cultural-3.jpg', 'user' => 'Manpreet G.', 'location' => 'Sydney'],
                    ['src' => 'gallery/cultural-4.jpg', 'user' => 'Hardeep B.', 'location' => 'Brisbane']
                ]
            ]
        ];

        return view('community.gallery', compact('galleries'));
    }

    /**
     * Customer spotlight page
     */
    public function spotlight()
    {
        $spotlights = [
            [
                'name' => 'Maya Patel',
                'location' => 'Melbourne',
                'story' => 'Maya is a graphic designer who found Edition By Punjab while searching for streetwear that reflected her cultural identity. She loves how our designs blend traditional Punjabi elements with contemporary fashion.',
                'favorite_piece' => 'Phulkari Inspired Hoodie',
                'style_tip' => 'I love layering cultural pieces with neutral basics. It creates a perfect balance between heritage and modern style.',
                'image' => 'spotlight/maya.jpg',
                'instagram' => '@maya_designs'
            ],
            [
                'name' => 'Jordan Kim',
                'location' => 'Sydney',
                'story' => 'Jordan is a sustainability advocate who appreciates our commitment to ethical production. As a content creator, he showcases how sustainable fashion can be stylish and impactful.',
                'favorite_piece' => 'Organic Cotton Tee',
                'style_tip' => 'Sustainable fashion doesn\'t mean compromising on style. Choose quality pieces that reflect your values and wear them with pride.',
                'image' => 'spotlight/jordan.jpg',
                'instagram' => '@sustainable_jordan'
            ]
        ];

        return view('community.spotlight', compact('spotlights'));
    }

    /**
     * Handle style submission form
     */
    public function store(Request $request)
    {
        $request->validate([
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:10240', // 10MB max
            'name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
            'caption' => 'required|string|min:50|max:1000',
            'products' => 'nullable|string|max:500',
            'instagram' => 'nullable|string|max:255',
            'consent_gallery' => 'required|accepted',
            'consent_social' => 'nullable|boolean',
            'consent_marketing' => 'nullable|boolean'
        ], [
            'photo.required' => 'Please upload a photo of your style.',
            'photo.image' => 'The uploaded file must be an image.',
            'photo.mimes' => 'Please upload a valid image file (JPEG, PNG, JPG, or GIF).',
            'photo.max' => 'The image size must not exceed 10MB.',
            'name.required' => 'Please enter your name.',
            'location.required' => 'Please select your location.',
            'caption.required' => 'Please tell us about your style.',
            'caption.min' => 'Please write at least 50 characters about your style.',
            'caption.max' => 'Caption must not exceed 1000 characters.',
            'consent_gallery.required' => 'You must consent to display your photo in the gallery.'
        ]);

        try {
            // Handle file upload
            if ($request->hasFile('photo')) {
                $photo = $request->file('photo');
                $filename = 'community_' . time() . '_' . uniqid() . '.' . $photo->getClientOriginalExtension();
                
                // Store in public/images/community directory
                $path = $photo->move(public_path('images/community'), $filename);
                $photoPath = 'community/' . $filename;
            }

            // Create the community post
            $communityPost = CommunityPost::create([
                'name' => $request->name,
                'location' => $request->location,
                'caption' => $request->caption,
                'products_featured' => $request->products,
                'instagram_handle' => $request->instagram,
                'image_path' => $photoPath ?? null,
                'consent_gallery' => $request->boolean('consent_gallery'),
                'consent_social' => $request->boolean('consent_social'),
                'consent_marketing' => $request->boolean('consent_marketing'),
                'status' => 'pending'
            ]);

            // In a real app, you might also:
            // 1. Send notification email to admin
            // 2. Queue image processing job
            // 3. Send thank you email to user
            
            return redirect()->route('community.index')
                ->with('success', 'Thank you for sharing your style! Your submission is under review and will appear in our gallery once approved.');

        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Something went wrong while submitting your style. Please try again.');
        }
    }
}