<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Testimonial;
use App\Models\Banner;
use App\Models\InstagramFeed;
use App\Models\Inquiry;
use App\Models\CompanyInfo;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    /**
     * Display the home page.
     */
    public function index()
    {
        // Get banners for hero slider and promotional sections
        $heroBanners = Banner::active()
                            ->hero()
                            ->orderBy('sort_order')
                            ->get();
        
        $promoBanners = Banner::active()
                              ->promo()
                              ->orderBy('position')
                              ->orderBy('sort_order')
                              ->get();

        // Get new slider banners
        $homeHeroSliders = Banner::active()
                                ->homeHeroSlider()
                                ->orderBy('sort_order')
                                ->get();

        $belowCategoriesSliders = Banner::active()
                                       ->belowCategoriesSlider()
                                       ->orderBy('sort_order')
                                       ->get();

        $trainSliders = Banner::active()
                              ->trainSlider()
                              ->orderBy('sort_order')
                              ->get();

        // Get Instagram feeds
        $instagramFeeds = InstagramFeed::active()
                                       ->orderBy('sort_order')
                                       ->orderBy('created_at', 'desc')
                                       ->take(8)
                                       ->get();
        
        // Get featured products
        $featuredProducts = Product::active()
                                  ->featured()
                                  ->inStock()
                                  ->with('category')
                                  ->take(8)
                                  ->get();
        
        // Get new arrivals
        $newArrivals = Product::active()
                             ->inStock()
                             ->with('category')
                             ->latest()
                             ->take(8)
                             ->get();
        
        // Get categories
        $categories = Category::active()
                            ->parents()
                            ->orderBy('sort_order')
                            ->take(6)
                            ->get();
        
        // Get testimonials
        $testimonials = Testimonial::active()
                                  ->latest()
                                  ->take(6)
                                  ->get();
        
        // Get company info
        $companyInfo = CompanyInfo::first();
        
        return view('home', compact(
            'heroBanners',
            'promoBanners',
            'homeHeroSliders',
            'belowCategoriesSliders',
            'trainSliders',
            'instagramFeeds',
            'featuredProducts',
            'newArrivals',
            'categories',
            'testimonials',
            'companyInfo'
        ));
    }

    /**
     * Display the about page.
     */
    public function about()
    {
        $testimonials = Testimonial::active()
                                  ->latest()
                                  ->get();
        
        $categories = Category::active()->get();
        $companyInfo = CompanyInfo::first();
        
        return view('about', compact('testimonials', 'categories', 'companyInfo'));
    }

    /**
     * Display the contact page.
     */
    public function contact()
    {
        $categories = Category::active()->get();
        $companyInfo = CompanyInfo::first();
        
        return view('contact', compact('categories', 'companyInfo'));
    }

    /**
     * Submit contact inquiry.
     */
    public function submitInquiry(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'company' => 'nullable|string|max:255',
            'country' => 'nullable|string|max:100',
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
            'product_id' => 'nullable|exists:products,id'
        ]);

        Inquiry::create($validated);

        return redirect()->back()->with('success', 'Thank you for your inquiry! We will get back to you soon.');
    }
}