<?php

// app/Http/Controllers/ShopController.php
namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\Wishlist;
use App\Models\DiscountCampaign;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ShopController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::with(['category', 'reviews'])
            ->where('is_active', true)
            ->where('stock', '>', 0);

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('brand', 'like', "%{$search}%");
            });
        }

        // Category filter
        if ($request->has('category')) {
            $category = Category::where('slug', $request->category)->first();
            if ($category) {
                $query->where('category_id', $category->id);
            }
        }

        // Collection filter
        if ($request->has('collection')) {
            switch($request->collection) {
                case 'new-arrivals':
                    $query->orderBy('created_at', 'desc')->take(20);
                    break;
                case 'best-sellers':
                    $query->orderBy('sold_count', 'desc')->take(20);
                    break;
                case 'sale':
                    // Show products from active discount campaigns using pivot table
                    $query->whereHas('activeDiscountCampaigns')
                          ->with(['activeDiscountCampaigns' => function($q) {
                              $q->withPivot('original_price', 'sale_price');
                          }]);
                    break;
                case 'punjab-edition':
                    $query->where('brand', 'Punjab Edition');
                    break;
            }
        }

        // Brand filter
        if ($request->has('brand')) {
            $query->where('brand', $request->brand);
        }

        // Price filter
        if ($request->has('min_price')) {
            $query->where('price', '>=', $request->min_price);
        }
        if ($request->has('max_price')) {
            $query->where('price', '<=', $request->max_price);
        }

        // Size filter
        if ($request->has('size')) {
            $query->whereJsonContains('sizes', $request->size);
        }

        // Color filter
        if ($request->has('color')) {
            $query->whereJsonContains('colors', $request->color);
        }

        // Sorting
        switch($request->get('sort', 'featured')) {
            case 'price_low':
                $query->orderBy('price', 'asc');
                break;
            case 'price_high':
                $query->orderBy('price', 'desc');
                break;
            case 'newest':
                $query->orderBy('created_at', 'desc');
                break;
            case 'best_selling':
                $query->orderBy('sold_count', 'desc');
                break;
            default:
                $query->orderBy('is_featured', 'desc')
                      ->orderBy('created_at', 'desc');
        }

        $products = $query->paginate(12);
        $categories = Category::where('is_active', true)->withCount('products')->get();
        
        // Get unique sizes, colors, and brands for filters
        $allProducts = Product::where('is_active', true)->get();
        $sizes = collect();
        $colors = collect();
        $brands = $allProducts->pluck('brand')->unique()->filter();
        
        foreach ($allProducts as $product) {
            if ($product->sizes) {
                $sizes = $sizes->merge(json_decode($product->sizes, true) ?? []);
            }
            if ($product->colors) {
                $colors = $colors->merge(json_decode($product->colors, true) ?? []);
            }
        }
        
        $sizes = $sizes->unique()->sort();
        $colors = $colors->unique()->sort();

        // Get wishlist items for authenticated users
        $wishlistItems = [];
        if (Auth::check()) {
            $wishlistItems = Wishlist::where('user_id', Auth::id())
                ->pluck('product_id')
                ->toArray();
        }

        return view('shop.index', compact(
            'products', 'categories', 'sizes', 'colors', 'brands', 'wishlistItems'
        ));
    }

    /**
     * Show discounted products for the active campaign
     */
    public function discounted(Request $request)
    {
        $campaign = DiscountCampaign::active()->first();

        if (!$campaign) {
            return redirect()->route('shop.index')
                ->with('info', 'No active discount campaigns at the moment.');
        }

        $query = Product::with('category')
            ->where('is_active', true);

        // Normalize product_ids and category_ids (handle JSON or array)
        $productIds = [];
        $categoryIds = [];

        if (!empty($campaign->product_ids)) {
            $productIds = is_array($campaign->product_ids)
                ? $campaign->product_ids
                : (json_decode($campaign->product_ids, true) ?? []);
        }

        if (!empty($campaign->category_ids)) {
            $categoryIds = is_array($campaign->category_ids)
                ? $campaign->category_ids
                : (json_decode($campaign->category_ids, true) ?? []);
        }

        // Apply campaign filters
        if (!empty($productIds)) {
            $query->whereIn('id', $productIds);
        } elseif (!empty($categoryIds)) {
            $query->whereIn('category_id', $categoryIds);
        } else {
            // If campaign has no specific filters, optionally limit to items with compare_price
            // $query->whereNotNull('compare_price')->whereColumn('compare_price', '>', 'price');
        }

        // Apply sorting
        if ($request->has('sort')) {
            switch ($request->sort) {
                case 'price_low':
                    $query->orderBy('price', 'asc');
                    break;
                case 'price_high':
                    $query->orderBy('price', 'desc');
                    break;
                case 'newest':
                    $query->latest();
                    break;
                case 'popular':
                    $query->orderBy('views', 'desc');
                    break;
                default:
                    $query->orderBy('is_featured', 'desc')->latest();
            }
        } else {
            $query->orderBy('is_featured', 'desc')->latest();
        }

        $products = $query->paginate(12);

        // Apply discount to products for display (non-persistent)
        foreach ($products as $product) {
            $product->original_price = $product->price;
            $product->discount_percentage = $campaign->discount_percentage;
            $product->discounted_price = round($product->price * (1 - $campaign->discount_percentage / 100), 2);
        }

        $categories = Category::where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        $sizes = collect();
        $colors = collect();
        $brands = collect();

        // Collect sizes/colors/brands from the filtered products (optional)
        foreach ($products as $p) {
            if ($p->sizes) {
                $sizes = $sizes->merge(json_decode($p->sizes, true) ?? []);
            }
            if ($p->colors) {
                $colors = $colors->merge(json_decode($p->colors, true) ?? []);
            }
            if ($p->brand) {
                $brands->push($p->brand);
            }
        }
        $sizes = $sizes->unique()->sort();
        $colors = $colors->unique()->sort();
        $brands = $brands->unique()->filter()->values();

        $wishlistItems = [];
        if (Auth::check()) {
            $wishlistItems = Wishlist::where('user_id', Auth::id())
                ->pluck('product_id')
                ->toArray();
        }

        return view('shop.discounted', compact(
            'products', 'categories', 'sizes', 'colors', 'brands',
            'wishlistItems', 'campaign'
        ));
    }

    public function show($slug)
    {
        $product = Product::with(['category', 'reviews.user'])
            ->where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        // Increment view count
        $product->increment('views');

        // Get related products
        $relatedProducts = Product::where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->where('is_active', true)
            ->where('stock', '>', 0)
            ->take(4)
            ->get();

        // Check if in wishlist
        $inWishlist = false;
        if (Auth::check()) {
            $inWishlist = Wishlist::where('user_id', Auth::id())
                ->where('product_id', $product->id)
                ->exists();
        }

        // Parse gallery images
        $galleryImages = [];
        if ($product->gallery_images) {
            $galleryImages = json_decode($product->gallery_images, true) ?? [];
        }

        // Parse sizes and colors
        $sizes = json_decode($product->sizes, true) ?? [];
        $colors = json_decode($product->colors, true) ?? [];

        return view('shop.product', compact(
            'product', 'relatedProducts', 'inWishlist', 'galleryImages', 'sizes', 'colors'
        ));
    }

    public function category($slug)
    {
        $category = Category::where('slug', $slug)
            ->where('is_active', true)
            ->firstOrFail();

        return $this->index(request()->merge(['category' => $slug]));
    }

    public function quickView($id)
    {
        $product = Product::with('category')
            ->where('id', $id)
            ->where('is_active', true)
            ->firstOrFail();

        $sizes = json_decode($product->sizes, true) ?? [];
        $colors = json_decode($product->colors, true) ?? [];

        return view('shop.quick-view', compact('product', 'sizes', 'colors'));
    }

    public function toggleWishlist(Request $request)
    {
        if (!Auth::check()) {
            return response()->json([
                'success' => false,
                'message' => 'Please login to add items to wishlist'
            ], 401);
        }

        $request->validate([
            'product_id' => 'required|exists:products,id'
        ]);

        $wishlist = Wishlist::where('user_id', Auth::id())
            ->where('product_id', $request->product_id)
            ->first();

        if ($wishlist) {
            $wishlist->delete();
            $added = false;
            $message = 'Removed from wishlist';
        } else {
            Wishlist::create([
                'user_id' => Auth::id(),
                'product_id' => $request->product_id
            ]);
            $added = true;
            $message = 'Added to wishlist';
        }

        return response()->json([
            'success' => true,
            'added' => $added,
            'message' => $message
        ]);
    }

    public function search(Request $request)
    {
        $request->validate([
            'q' => 'required|string|min:2'
        ]);

        $query = $request->q;
        
        $products = Product::where('is_active', true)
            ->where('stock', '>', 0)
            ->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('description', 'like', "%{$query}%")
                  ->orWhere('sku', 'like', "%{$query}%")
                  ->orWhere('brand', 'like', "%{$query}%");
            })
            ->take(10)
            ->get(['id', 'name', 'slug', 'featured_image', 'price']);

        return response()->json($products);
    }
}