<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Models\Product;
use App\Models\Category;
use App\Models\Blog;

class SitemapController extends Controller
{
    /**
     * Main sitemap index
     */
    public function index()
    {
        $sitemaps = [
            [
                'url' => route('sitemap.static'),
                'lastmod' => now()->toISOString()
            ],
            [
                'url' => route('sitemap.products'),
                'lastmod' => Product::latest('updated_at')->first()?->updated_at?->toISOString() ?? now()->toISOString()
            ],
            [
                'url' => route('sitemap.categories'),
                'lastmod' => Category::latest('updated_at')->first()?->updated_at?->toISOString() ?? now()->toISOString()
            ],
            [
                'url' => route('sitemap.blogs'),
                'lastmod' => Blog::latest('updated_at')->first()?->updated_at?->toISOString() ?? now()->toISOString()
            ]
        ];

        return response()->view('sitemaps.index', compact('sitemaps'))
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Static pages sitemap
     */
    public function static()
    {
        $urls = [
            [
                'url' => route('home'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'daily',
                'priority' => '1.0'
            ],
            [
                'url' => route('shop.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'daily',
                'priority' => '0.9'
            ],
            [
                'url' => route('blogs.index'),
                'lastmod' => Blog::latest('updated_at')->first()?->updated_at?->toISOString() ?? now()->toISOString(),
                'changefreq' => 'daily',
                'priority' => '0.8'
            ],
            [
                'url' => route('about'),
                'lastmod' => now()->subDays(7)->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.6'
            ],
            [
                'url' => route('contact'),
                'lastmod' => now()->subDays(7)->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.6'
            ],
            // Lookbook pages
            [
                'url' => route('lookbook.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            // Style guide pages
            [
                'url' => route('style-guide.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            // Community pages
            [
                'url' => route('community.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => route('community.gallery'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.6'
            ],
            [
                'url' => route('community.spotlight'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.6'
            ],
            // Trends pages
            [
                'url' => route('trends.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'daily',
                'priority' => '0.8'
            ],
            [
                'url' => route('trends.weekly'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => route('trends.seasonal'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.6'
            ],
            [
                'url' => route('trends.news'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'daily',
                'priority' => '0.7'
            ],
            // Size guide pages
            [
                'url' => route('size-guide.index'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.7'
            ],
            [
                'url' => route('size-guide.finder'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.6'
            ],
            // City pages - High priority for local SEO
            [
                'url' => route('city.melbourne'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.9'
            ],
            [
                'url' => route('city.sydney'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'weekly',
                'priority' => '0.9'
            ],
            // Long-tail keyword pages - High priority for SEO
            [
                'url' => route('where-to-buy-streetwear'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.9'
            ],
            [
                'url' => route('punjabi-fusion-streetwear'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.9'
            ],
            // FAQ page
            [
                'url' => route('faq'),
                'lastmod' => now()->toISOString(),
                'changefreq' => 'monthly',
                'priority' => '0.8'
            ]
        ];

        return response()->view('sitemaps.urlset', compact('urls'))
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Products sitemap
     */
    public function products()
    {
        $urls = Product::where('is_active', true)
            ->select('id', 'slug', 'updated_at', 'featured_image')
            ->chunk(1000, function ($products) use (&$urls) {
                foreach ($products as $product) {
                    $urls[] = [
                        'url' => route('shop.product', $product->slug),
                        'lastmod' => $product->updated_at->toISOString(),
                        'changefreq' => 'weekly',
                        'priority' => '0.8',
                        'images' => $this->getProductImages($product)
                    ];
                }
            });

        return response()->view('sitemaps.urlset', compact('urls'))
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Categories sitemap
     */
    public function categories()
    {
        $urls = [];
        
        Category::where('is_active', true)
            ->select('id', 'slug', 'updated_at', 'image')
            ->get()
            ->each(function ($category) use (&$urls) {
                $urls[] = [
                    'url' => route('shop.category', $category->slug),
                    'lastmod' => $category->updated_at->toISOString(),
                    'changefreq' => 'weekly',
                    'priority' => '0.7',
                    'images' => $category->image ? [['url' => asset('storage/' . $category->image)]] : []
                ];
            });

        return response()->view('sitemaps.urlset', compact('urls'))
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Blogs sitemap
     */
    public function blogs()
    {
        $urls = [];
        
        Blog::where('is_published', true)
            ->select('slug', 'updated_at', 'featured_image', 'title')
            ->get()
            ->each(function ($blog) use (&$urls) {
                $urls[] = [
                    'url' => route('blogs.show', $blog->slug),
                    'lastmod' => $blog->updated_at->toISOString(),
                    'changefreq' => 'monthly',
                    'priority' => '0.6',
                    'images' => $blog->featured_image ? [['url' => asset('storage/' . $blog->featured_image)]] : []
                ];
            });

        return response()->view('sitemaps.urlset', compact('urls'))
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Get product images for sitemap
     */
    private function getProductImages($product)
    {
        $images = [];
        
        if ($product->featured_image) {
            $images[] = [
                'url' => asset('storage/' . $product->featured_image),
                'caption' => $product->name ?? 'Product Image'
            ];
        }
        
        return $images;
    }
}