<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;

class SizeGuideController extends Controller
{
    /**
     * Size guide main page
     */
    public function index()
    {
        $categories = Category::where('is_active', true)->get();
        
        $sizeGuides = [
            'hoodies' => [
                'name' => 'Hoodies & Sweatshirts',
                'description' => 'Find your perfect fit for our premium hoodies and sweatshirts',
                'image' => 'size-guides/hoodie-guide.jpg',
                'fit_type' => 'Relaxed/Oversized'
            ],
            'tees' => [
                'name' => 'T-Shirts & Tees',
                'description' => 'Sizing information for our comfortable and stylish tees',
                'image' => 'size-guides/tee-guide.jpg',
                'fit_type' => 'Regular/Relaxed'
            ],
            'bottoms' => [
                'name' => 'Bottoms & Joggers',
                'description' => 'Get the right fit for joggers, pants, and shorts',
                'image' => 'size-guides/bottoms-guide.jpg',
                'fit_type' => 'Regular/Tapered'
            ],
            'accessories' => [
                'name' => 'Accessories & Caps',
                'description' => 'Size information for hats, caps, and accessories',
                'image' => 'size-guides/accessories-guide.jpg',
                'fit_type' => 'One Size/Adjustable'
            ]
        ];

        return view('size-guide.index', compact('sizeGuides', 'categories'));
    }

    /**
     * Specific category size guide
     */
    public function show($category)
    {
        $sizeCharts = [
            'hoodies' => [
                'name' => 'Hoodies & Sweatshirts Size Guide',
                'meta_title' => 'Hoodie Size Guide Australia | Edition By Punjab Sizing Chart',
                'meta_description' => 'Find your perfect hoodie fit with our comprehensive size guide. Australian sizing chart for premium streetwear hoodies and sweatshirts.',
                'keywords' => 'hoodie size guide, sweatshirt sizing, streetwear size chart, australian clothing sizes, hoodie fit guide',
                'fit_description' => 'Our hoodies feature a relaxed, comfortable fit with slightly dropped shoulders for that perfect streetwear silhouette.',
                'size_chart' => [
                    'S' => ['chest' => '102cm', 'length' => '69cm', 'shoulder' => '58cm', 'sleeve' => '60cm'],
                    'M' => ['chest' => '108cm', 'length' => '71cm', 'shoulder' => '60cm', 'sleeve' => '62cm'],
                    'L' => ['chest' => '114cm', 'length' => '73cm', 'shoulder' => '62cm', 'sleeve' => '64cm'],
                    'XL' => ['chest' => '120cm', 'length' => '75cm', 'shoulder' => '64cm', 'sleeve' => '66cm'],
                    'XXL' => ['chest' => '126cm', 'length' => '77cm', 'shoulder' => '66cm', 'sleeve' => '68cm']
                ],
                'sizing_tips' => [
                    'For a relaxed fit, choose your regular size',
                    'For an oversized look, go one size up',
                    'Consider shrinkage - our hoodies may shrink 2-3cm after first wash',
                    'Check individual product pages for specific fit notes'
                ],
                'care_instructions' => [
                    'Wash in cold water to prevent shrinking',
                    'Air dry when possible to maintain shape',
                    'Iron inside out to protect prints',
                    'Avoid fabric softener on first few washes'
                ]
            ],
            'tees' => [
                'name' => 'T-Shirts & Tees Size Guide',
                'meta_title' => 'T-Shirt Size Guide Australia | Streetwear Tee Sizing Chart',
                'meta_description' => 'Perfect fit guaranteed with our t-shirt size guide. Australian streetwear tee sizing chart with detailed measurements for all body types.',
                'keywords' => 't-shirt size guide, tee sizing chart, streetwear t-shirt sizes, australian tee sizes, t-shirt fit guide',
                'fit_description' => 'Our tees offer a comfortable regular fit with quality cotton construction for everyday wear.',
                'size_chart' => [
                    'XS' => ['chest' => '92cm', 'length' => '66cm', 'shoulder' => '42cm', 'sleeve' => '19cm'],
                    'S' => ['chest' => '98cm', 'length' => '69cm', 'shoulder' => '44cm', 'sleeve' => '20cm'],
                    'M' => ['chest' => '104cm', 'length' => '72cm', 'shoulder' => '46cm', 'sleeve' => '21cm'],
                    'L' => ['chest' => '110cm', 'length' => '75cm', 'shoulder' => '48cm', 'sleeve' => '22cm'],
                    'XL' => ['chest' => '116cm', 'length' => '78cm', 'shoulder' => '50cm', 'sleeve' => '23cm'],
                    'XXL' => ['chest' => '122cm', 'length' => '81cm', 'shoulder' => '52cm', 'sleeve' => '24cm']
                ],
                'sizing_tips' => [
                    'Regular fit - true to size for most people',
                    'Pre-shrunk cotton minimizes size change after washing',
                    'Check product details for specific fits (slim, regular, relaxed)',
                    'Model measurements provided on each product page'
                ],
                'care_instructions' => [
                    'Machine wash cold with like colors',
                    'Tumble dry low or air dry',
                    'Iron on medium heat if needed',
                    'Turn inside out when washing to protect prints'
                ]
            ],
            'bottoms' => [
                'name' => 'Bottoms & Joggers Size Guide',
                'meta_title' => 'Joggers & Pants Size Guide | Australian Streetwear Bottoms Sizing',
                'meta_description' => 'Find the perfect fit for streetwear joggers and pants. Comprehensive Australian sizing guide for comfortable, stylish bottoms.',
                'keywords' => 'joggers size guide, pants sizing chart, streetwear bottoms sizes, australian pants sizes, joggers fit guide',
                'fit_description' => 'Our bottoms feature a comfortable regular fit with tapered legs for a modern streetwear silhouette.',
                'size_chart' => [
                    'S' => ['waist' => '76cm', 'hip' => '102cm', 'inseam' => '76cm', 'leg_opening' => '28cm'],
                    'M' => ['waist' => '81cm', 'hip' => '107cm', 'inseam' => '78cm', 'leg_opening' => '29cm'],
                    'L' => ['waist' => '86cm', 'hip' => '112cm', 'inseam' => '80cm', 'leg_opening' => '30cm'],
                    'XL' => ['waist' => '91cm', 'hip' => '117cm', 'inseam' => '82cm', 'leg_opening' => '31cm'],
                    'XXL' => ['waist' => '96cm', 'hip' => '122cm', 'inseam' => '84cm', 'leg_opening' => '32cm']
                ],
                'sizing_tips' => [
                    'Elasticated waist provides comfortable fit',
                    'Measure at natural waistline for best fit',
                    'Consider length preference - can be worn cuffed or uncuffed',
                    'Drawstring allows for adjustment'
                ]
            ],
            'accessories' => [
                'name' => 'Accessories & Caps Size Guide',
                'meta_title' => 'Cap & Accessories Size Guide | Australian Streetwear Accessories',
                'meta_description' => 'Size guide for caps, hats, and streetwear accessories. Find the perfect fit for all your streetwear accessory needs.',
                'keywords' => 'cap size guide, hat sizing, streetwear accessories sizes, cap fit guide australia',
                'fit_description' => 'Most of our accessories feature adjustable sizing for comfort and versatility.',
                'size_info' => [
                    'caps' => [
                        'type' => 'Snapback/Adjustable',
                        'fit_range' => '54cm - 62cm head circumference',
                        'features' => ['Adjustable snapback closure', 'Curved or flat bill options', 'Structured crown']
                    ],
                    'beanies' => [
                        'type' => 'One Size Fits Most',
                        'fit_range' => '52cm - 60cm head circumference',
                        'features' => ['Stretchy knit material', 'Cuffed or uncuffed styling', 'Soft cotton blend']
                    ],
                    'bags' => [
                        'type' => 'Various Sizes',
                        'options' => ['Small (tote/crossbody)', 'Medium (backpack)', 'Large (duffle)'],
                        'measurements' => 'Listed on individual product pages'
                    ]
                ],
                'sizing_tips' => [
                    'Most caps are adjustable and fit head sizes 54-62cm',
                    'Beanies are stretchy and accommodate most head sizes',
                    'Check individual product pages for specific measurements',
                    'Consider hair volume when choosing beanie size'
                ]
            ]
        ];

        $guide = $sizeCharts[$category] ?? abort(404);

        return view('size-guide.show', compact('guide', 'category'));
    }

    /**
     * Size finder tool
     */
    public function finder()
    {
        return view('size-guide.finder');
    }

    /**
     * Returns to size based on measurements (AJAX)
     */
    public function findSize(Request $request)
    {
        $measurements = $request->validate([
            'category' => 'required|string',
            'chest' => 'nullable|numeric',
            'waist' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'weight' => 'nullable|numeric'
        ]);

        // Basic size recommendation logic
        $recommendations = $this->calculateSizeRecommendation($measurements);

        return response()->json($recommendations);
    }

    private function calculateSizeRecommendation($measurements)
    {
        // Simplified size recommendation logic
        $category = $measurements['category'];
        $chest = $measurements['chest'] ?? 0;
        $waist = $measurements['waist'] ?? 0;

        switch ($category) {
            case 'hoodies':
            case 'tees':
                if ($chest <= 96) return ['size' => 'S', 'confidence' => 85];
                if ($chest <= 102) return ['size' => 'M', 'confidence' => 90];
                if ($chest <= 108) return ['size' => 'L', 'confidence' => 90];
                if ($chest <= 114) return ['size' => 'XL', 'confidence' => 85];
                return ['size' => 'XXL', 'confidence' => 80];

            case 'bottoms':
                if ($waist <= 78) return ['size' => 'S', 'confidence' => 85];
                if ($waist <= 83) return ['size' => 'M', 'confidence' => 90];
                if ($waist <= 88) return ['size' => 'L', 'confidence' => 90];
                if ($waist <= 93) return ['size' => 'XL', 'confidence' => 85];
                return ['size' => 'XXL', 'confidence' => 80];

            default:
                return ['size' => 'M', 'confidence' => 70];
        }
    }
}