<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\StyleGuide;

class StyleGuideController extends Controller
{
    /**
     * Style guides main page
     */
    public function index()
    {
        // Try to get from database first
        $dbStyleGuides = StyleGuide::active()->ordered()->get();
        
        if ($dbStyleGuides->count() > 0) {
            // Use database style guides
            $cities = $dbStyleGuides->map(function ($guide) {
                return [
                    'name' => $guide->name,
                    'slug' => $guide->slug,
                    'tagline' => $guide->tagline,
                    'description' => $guide->description,
                    'image' => $guide->hero_image,
                    'color' => $guide->color_palette[0]['hex'] ?? '#333333'
                ];
            })->toArray();
        } else {
            // Fallback to hardcoded data
            $cities = [
            [
                'name' => 'Melbourne',
                'slug' => 'melbourne',
                'tagline' => 'Creative Capital Style',
                'description' => 'Artistic, layered, and experimental fashion inspired by Melbourne\'s creative culture',
                'image' => 'style-guides/melbourne.jpg',
                'color' => '#2C3E50'
            ],
            [
                'name' => 'Sydney',
                'slug' => 'sydney',
                'tagline' => 'Coastal Cool Vibes',
                'description' => 'Effortless beach-to-city style that embodies Sydney\'s relaxed sophistication',
                'image' => 'style-guides/sydney.jpg',
                'color' => '#3498DB'
            ],
            [
                'name' => 'Brisbane',
                'slug' => 'brisbane',
                'tagline' => 'Tropical Streetwear',
                'description' => 'Breathable, bright, and bold fashion perfect for Queensland\'s sunny climate',
                'image' => 'style-guides/brisbane.jpg',
                'color' => '#E67E22'
            ],
            [
                'name' => 'Perth',
                'slug' => 'perth',
                'tagline' => 'West Coast Edge',
                'description' => 'Bold, independent style that reflects Perth\'s unique coastal desert culture',
                'image' => 'style-guides/perth.jpg',
                'color' => '#9B59B6'
            ],
            [
                'name' => 'Adelaide',
                'slug' => 'adelaide',
                'tagline' => 'Festival Fashion',
                'description' => 'Creative, cultural, and community-focused style inspired by Adelaide\'s arts scene',
                'image' => 'style-guides/adelaide.jpg',
                'color' => '#E74C3C'
            ]
        ];
        }

        return view('style-guide.index', compact('cities'));
    }

    /**
     * Individual city style guide
     */
    public function show($city)
    {
        // Try to get from database first
        $dbGuide = StyleGuide::where('slug', $city)->where('is_active', true)->first();
        
        if ($dbGuide) {
            $guide = [
                'name' => $dbGuide->name,
                'tagline' => $dbGuide->tagline,
                'description' => $dbGuide->description,
                'region' => $dbGuide->region,
                'population' => $dbGuide->population,
                'fashion_score' => $dbGuide->fashion_score,
                'streetwear_popularity' => $dbGuide->streetwear_popularity,
                'meta_title' => $dbGuide->meta_title,
                'meta_description' => $dbGuide->meta_description,
                'keywords' => $dbGuide->keywords,
                'climate' => $dbGuide->climate,
                'vibe' => $dbGuide->vibe,
                'hero_image' => $dbGuide->hero_image,
                'content' => $dbGuide->content,
                'featured_products' => Product::where('is_featured', true)->limit(6)->get(),
                'styling_tips' => $dbGuide->styling_tips ?? [],
                'style_characteristics' => $dbGuide->style_characteristics ?? [],
                'key_pieces' => $dbGuide->key_pieces ?? [],
                'seasonal_styling' => $dbGuide->seasonal_styling ?? [],
                'cultural_influences' => $dbGuide->cultural_influences ?? [],
                'color_palette' => $dbGuide->color_palette ?? [],
                'shopping_areas' => $dbGuide->shopping_areas ?? [],
                'weather' => $dbGuide->weather ?? [],
                'style_gallery' => $dbGuide->style_gallery ?? []
            ];
        } else {
            // Fallback to hardcoded data
            $guides = [
            'melbourne' => [
                'name' => 'Melbourne',
                'tagline' => 'Creative Capital Style',
                'description' => 'Artistic, layered, and experimental fashion inspired by Melbourne\'s creative culture',
                'region' => 'Victoria',
                'population' => '5.2M',
                'fashion_score' => 9,
                'streetwear_popularity' => 85,
                'meta_title' => 'Melbourne Streetwear Style Guide | Creative Fashion Tips',
                'meta_description' => 'Discover Melbourne streetwear style with our comprehensive guide. Artistic layering, vintage mixing, and creative fashion tips for Melbourne\'s unique style.',
                'keywords' => 'melbourne streetwear, melbourne fashion, creative style melbourne, australian streetwear melbourne, melbourne style guide',
                'climate' => 'Variable - layers essential',
                'vibe' => 'Artistic, Creative, Experimental',
                'hero_image' => 'style-guides/melbourne-hero.jpg',
                'content' => $this->getMelbourneContent(),
                'featured_products' => Product::where('is_featured', true)->limit(6)->get(),
                'styling_tips' => [
                    'Master the art of layering for Melbourne\'s unpredictable weather',
                    'Mix vintage finds with contemporary pieces',
                    'Embrace artistic prints and unique silhouettes',
                    'Black is always a safe choice for Melbourne\'s creative scene'
                ],
                'style_characteristics' => [
                    ['title' => 'All-Black Everything', 'description' => 'Melbourne\'s unofficial uniform. Black provides the perfect canvas for creative expression.'],
                    ['title' => 'Layering Mastery', 'description' => 'Essential skill for Melbourne\'s "four seasons in one day" weather patterns.'],
                    ['title' => 'Artistic Expression', 'description' => 'Bold prints, unique silhouettes, and pieces that tell a creative story.']
                ],
                'key_pieces' => ['Oversized Hoodie', 'Denim Jacket', 'Black Skinny Jeans', 'Statement Sneakers', 'Artistic Tee', 'Layering Cardigan'],
                'seasonal_styling' => [
                    'spring' => ['season' => 'Spring', 'icon' => 'seedling', 'description' => 'Light layers and transitional pieces', 'tips' => ['Denim jackets over hoodies', 'Comfortable sneakers', 'Light scarves for style']],
                    'summer' => ['season' => 'Summer', 'icon' => 'sun', 'description' => 'Breathable fabrics with style', 'tips' => ['Cotton tees in artistic prints', 'Lightweight fabrics', 'Sun protection accessories']],
                    'autumn' => ['season' => 'Autumn', 'icon' => 'leaf', 'description' => 'Perfect hoodie weather', 'tips' => ['Cozy textures', 'Earth tones', 'Layered looks']],
                    'winter' => ['season' => 'Winter', 'icon' => 'snowflake', 'description' => 'Serious layering time', 'tips' => ['Heavy hoodies', 'Thermal layers', 'Statement outerwear']]
                ],
                'cultural_influences' => [
                    ['aspect' => 'Coffee Culture', 'description' => 'Melbourne\'s cafe scene influences relaxed, artisanal fashion choices'],
                    ['aspect' => 'Street Art', 'description' => 'Vibrant laneways inspire bold graphics and artistic expression'],
                    ['aspect' => 'Music Scene', 'description' => 'Independent music culture drives authentic, non-conformist style choices']
                ],
                'color_palette' => [
                    ['name' => 'Melbourne Black', 'hex' => '#000000'],
                    ['name' => 'Charcoal', 'hex' => '#36454F'],
                    ['name' => 'Burgundy', 'hex' => '#800020'],
                    ['name' => 'Forest Green', 'hex' => '#228B22']
                ],
                'shopping_areas' => [
                    ['name' => 'Chapel Street', 'description' => 'Eclectic mix of vintage and contemporary streetwear'],
                    ['name' => 'Fitzroy', 'description' => 'Independent designers and unique boutiques'],
                    ['name' => 'CBD Laneways', 'description' => 'Hidden gems and emerging local brands']
                ],
                'weather' => [
                    'avg_temp' => '15-20°C',
                    'rainfall' => 'Moderate, year-round',
                    'best_months' => 'Mar-May, Sep-Nov',
                    'style_note' => 'Always be prepared for sudden weather changes with versatile layers.'
                ],
                'style_gallery' => [
                    ['image' => 'gallery/melbourne-1.jpg', 'description' => 'Classic Melbourne all-black', 'style_type' => 'Minimalist'],
                    ['image' => 'gallery/melbourne-2.jpg', 'description' => 'Artistic layering', 'style_type' => 'Creative'],
                    ['image' => 'gallery/melbourne-3.jpg', 'description' => 'Street art inspired', 'style_type' => 'Urban'],
                    ['image' => 'gallery/melbourne-4.jpg', 'description' => 'Coffee culture casual', 'style_type' => 'Relaxed']
                ]
            ],
            'sydney' => [
                'name' => 'Sydney',
                'tagline' => 'Coastal Cool Vibes',
                'description' => 'Effortless beach-to-city style that embodies Sydney\'s relaxed sophistication',
                'region' => 'New South Wales',
                'population' => '5.3M',
                'fashion_score' => 8,
                'streetwear_popularity' => 78,
                'meta_title' => 'Sydney Streetwear Style Guide | Beach to City Fashion',
                'meta_description' => 'Master Sydney streetwear style with our guide to coastal cool fashion. From beach to city, discover how to dress for Sydney\'s lifestyle.',
                'keywords' => 'sydney streetwear, sydney fashion, coastal fashion, beach style sydney, sydney style guide',
                'climate' => 'Temperate - versatile pieces',
                'vibe' => 'Relaxed, Sophisticated, Coastal',
                'hero_image' => 'style-guides/sydney-hero.jpg',
                'content' => $this->getSydneyContent(),
                'featured_products' => Product::where('is_active', true)->limit(6)->get(),
                'styling_tips' => [
                    'Choose pieces that work from beach to boardroom',
                    'Invest in quality basics in neutral tones',
                    'Embrace the effortless sophistication Sydney is known for',
                    'Don\'t forget sun protection - style and safety combined'
                ],
                'style_characteristics' => [
                    ['title' => 'Beach to City', 'description' => 'Versatile pieces that transition seamlessly from harbor walks to business meetings.'],
                    ['title' => 'Quality Basics', 'description' => 'Investment pieces in premium materials and timeless cuts.'],
                    ['title' => 'Effortless Sophistication', 'description' => 'Looking polished without trying too hard - the Sydney way.']
                ],
                'key_pieces' => ['Premium Cotton Tee', 'Classic Denim', 'Clean White Sneakers', 'Light Jacket', 'Quality Hoodie', 'Sun-Smart Accessories'],
                'seasonal_styling' => [
                    'spring' => ['season' => 'Spring', 'icon' => 'seedling', 'description' => 'Perfect weather for layering', 'tips' => ['Light jackets for evening', 'Breathable cottons', 'Comfortable walking shoes']],
                    'summer' => ['season' => 'Summer', 'icon' => 'sun', 'description' => 'Beach-ready with city polish', 'tips' => ['UV protection built-in', 'Moisture-wicking fabrics', 'Easy beach-to-bar transitions']],
                    'autumn' => ['season' => 'Autumn', 'icon' => 'leaf', 'description' => 'Ideal conditions for streetwear', 'tips' => ['Perfect hoodie weather', 'Layered textures', 'Comfortable all-day wear']],
                    'winter' => ['season' => 'Winter', 'icon' => 'cloud-rain', 'description' => 'Mild winter styling', 'tips' => ['Light layering sufficient', 'Water-resistant pieces', 'Versatile wardrobe staples']]
                ],
                'cultural_influences' => [
                    ['aspect' => 'Harbor Culture', 'description' => 'Waterfront lifestyle inspires relaxed yet polished fashion choices'],
                    ['aspect' => 'Beach Lifestyle', 'description' => 'Coastal living demands versatile pieces that work for multiple occasions'],
                    ['aspect' => 'Business District', 'description' => 'CBD influence brings elevated basics and smart-casual elements']
                ],
                'color_palette' => [
                    ['name' => 'Ocean Blue', 'hex' => '#006994'],
                    ['name' => 'Sand Beige', 'hex' => '#F5E6D3'],
                    ['name' => 'Harbor Grey', 'hex' => '#708090'],
                    ['name' => 'Sunset Orange', 'hex' => '#FF8C69']
                ],
                'shopping_areas' => [
                    ['name' => 'Surry Hills', 'description' => 'Independent boutiques and creative streetwear brands'],
                    ['name' => 'Newtown', 'description' => 'Alternative fashion and vintage finds'],
                    ['name' => 'Bondi Junction', 'description' => 'Beach-to-city fashion and activewear specialists']
                ],
                'weather' => [
                    'avg_temp' => '18-23°C',
                    'rainfall' => 'Moderate, winter peaks',
                    'best_months' => 'Mar-May, Sep-Nov',
                    'style_note' => 'Mild climate allows for year-round versatile styling with light layers.'
                ],
                'style_gallery' => [
                    ['image' => 'gallery/sydney-1.jpg', 'description' => 'Beach to city transition', 'style_type' => 'Versatile'],
                    ['image' => 'gallery/sydney-2.jpg', 'description' => 'Harbor walk ready', 'style_type' => 'Casual'],
                    ['image' => 'gallery/sydney-3.jpg', 'description' => 'Effortless sophistication', 'style_type' => 'Smart-Casual'],
                    ['image' => 'gallery/sydney-4.jpg', 'description' => 'Coastal cool vibes', 'style_type' => 'Relaxed']
                ]
            ],
            'brisbane' => [
                'name' => 'Brisbane',
                'tagline' => 'Tropical Streetwear',
                'description' => 'Breathable, bright, and bold fashion perfect for Queensland\'s sunny climate',
                'region' => 'Queensland',
                'population' => '2.6M',
                'fashion_score' => 7,
                'streetwear_popularity' => 72,
                'meta_title' => 'Brisbane Streetwear Style Guide | Tropical Fashion Tips',
                'meta_description' => 'Navigate Brisbane\'s tropical climate with style. Our guide to breathable streetwear, bold colors, and comfortable fashion for Queensland living.',
                'keywords' => 'brisbane streetwear, brisbane fashion, tropical streetwear, queensland fashion, brisbane style guide',
                'climate' => 'Subtropical - breathable fabrics',
                'vibe' => 'Relaxed, Colorful, Comfortable',
                'hero_image' => 'style-guides/brisbane-hero.jpg',
                'content' => $this->getBrisbaneContent(),
                'featured_products' => Product::where('is_active', true)->limit(6)->get(),
                'styling_tips' => [
                    'Prioritize breathable, moisture-wicking fabrics',
                    'Don\'t shy away from bold tropical colors',
                    'Lightweight layers for air-conditioned indoor spaces',
                    'Comfortable footwear is essential for Brisbane\'s walking culture'
                ],
                'style_characteristics' => [
                    ['title' => 'Climate-Smart', 'description' => 'Breathable fabrics and UV protection built into everyday pieces.'],
                    ['title' => 'Bold Colors', 'description' => 'Tropical hues that reflect Queensland\'s vibrant outdoor lifestyle.'],
                    ['title' => 'Comfort First', 'description' => 'Functional fashion that works in heat and humidity.']
                ],
                'key_pieces' => ['Breathable Tee', 'Lightweight Shorts', 'UV-Protection Hoodie', 'Comfortable Sneakers', 'Sun Hat', 'Quick-Dry Fabrics'],
                'seasonal_styling' => [
                    'wet_season' => ['season' => 'Wet Season', 'icon' => 'cloud-rain', 'description' => 'Nov-Apr: Hot and humid', 'tips' => ['Quick-dry fabrics', 'Covered footwear', 'Lightweight rain protection']],
                    'dry_season' => ['season' => 'Dry Season', 'icon' => 'sun', 'description' => 'May-Oct: Perfect weather', 'tips' => ['Light layering possible', 'Comfortable all-day wear', 'Sun protection essential']]
                ],
                'cultural_influences' => [
                    ['aspect' => 'Outdoor Lifestyle', 'description' => 'Year-round outdoor activities influence practical, comfortable fashion choices'],
                    ['aspect' => 'River Culture', 'description' => 'Brisbane River lifestyle promotes relaxed, water-friendly fashion'],
                    ['aspect' => 'Subtropical Living', 'description' => 'Hot climate demands breathable, functional yet stylish clothing']
                ],
                'color_palette' => [
                    ['name' => 'Tropical Coral', 'hex' => '#FF7F7F'],
                    ['name' => 'Sunshine Yellow', 'hex' => '#FFD700'],
                    ['name' => 'Palm Green', 'hex' => '#32CD32'],
                    ['name' => 'Sky Blue', 'hex' => '#87CEEB']
                ],
                'shopping_areas' => [
                    ['name' => 'Fortitude Valley', 'description' => 'Alternative fashion and streetwear boutiques'],
                    ['name' => 'West End', 'description' => 'Sustainable and locally-made fashion'],
                    ['name' => 'Queen Street Mall', 'description' => 'Mainstream and activewear brands']
                ],
                'weather' => [
                    'avg_temp' => '20-28°C',
                    'rainfall' => 'High in summer, low in winter',
                    'best_months' => 'Apr-Oct',
                    'style_note' => 'Hot, humid summers require strategic fabric choices and breathable styling.'
                ],
                'style_gallery' => [
                    ['image' => 'gallery/brisbane-1.jpg', 'description' => 'Tropical color blocking', 'style_type' => 'Bold'],
                    ['image' => 'gallery/brisbane-2.jpg', 'description' => 'Climate-smart casual', 'style_type' => 'Practical'],
                    ['image' => 'gallery/brisbane-3.jpg', 'description' => 'River walk ready', 'style_type' => 'Active'],
                    ['image' => 'gallery/brisbane-4.jpg', 'description' => 'Subtropical comfort', 'style_type' => 'Relaxed']
                ]
            ],
            'perth' => [
                'name' => 'Perth',
                'tagline' => 'West Coast Edge',
                'description' => 'Bold, independent style that reflects Perth\'s unique coastal desert culture',
                'region' => 'Western Australia',
                'population' => '2.1M',
                'fashion_score' => 7,
                'streetwear_popularity' => 75,
                'meta_title' => 'Perth Streetwear Style Guide | West Coast Fashion',
                'meta_description' => 'Discover Perth\'s unique streetwear style. Bold, independent fashion that reflects Western Australia\'s coastal desert culture.',
                'keywords' => 'perth streetwear, perth fashion, west coast style, western australia fashion, perth style guide',
                'climate' => 'Mediterranean - versatile styling',
                'vibe' => 'Independent, Bold, Relaxed',
                'hero_image' => 'style-guides/perth-hero.jpg',
                'content' => $this->getPerthContent(),
                'featured_products' => Product::where('is_featured', true)->limit(6)->get(),
                'styling_tips' => [
                    'Embrace Perth\'s independent fashion spirit',
                    'Bold colors and patterns reflect the vibrant landscape',
                    'Comfortable yet stylish for Perth\'s outdoor lifestyle',
                    'Sun protection is stylish and essential'
                ],
                'style_characteristics' => [
                    ['title' => 'Independent Spirit', 'description' => 'Perth fashion creates its own rules, separate from east coast trends.'],
                    ['title' => 'Landscape Inspired', 'description' => 'Desert and ocean contrasts influence bold color choices.'],
                    ['title' => 'Sun-Smart Style', 'description' => 'UV protection integrated into everyday fashion choices.']
                ],
                'key_pieces' => ['Bold Graphic Tee', 'Sun-Smart Hoodie', 'Comfortable Cargo Shorts', 'Sturdy Sneakers', 'Wide-Brim Hat', 'Independent Brand Pieces'],
                'seasonal_styling' => [
                    'summer' => ['season' => 'Summer', 'icon' => 'sun', 'description' => 'Hot, dry summers', 'tips' => ['Light colors reflect heat', 'UV protection essential', 'Breathable natural fabrics']],
                    'winter' => ['season' => 'Winter', 'icon' => 'cloud', 'description' => 'Mild, wet winters', 'tips' => ['Light layering sufficient', 'Water-resistant pieces', 'Comfortable indoor-outdoor wear']],
                    'spring' => ['season' => 'Spring', 'icon' => 'seedling', 'description' => 'Perfect weather', 'tips' => ['Ideal for any style', 'Mix textures freely', 'Outdoor adventure ready']],
                    'autumn' => ['season' => 'Autumn', 'icon' => 'leaf', 'description' => 'Extended summer', 'tips' => ['Continued sun protection', 'Gradual layering', 'Versatile all-day pieces']]
                ],
                'cultural_influences' => [
                    ['aspect' => 'Isolation Pride', 'description' => 'Geographic isolation fosters independent, unique fashion choices'],
                    ['aspect' => 'Mining Culture', 'description' => 'Practical, durable fashion influenced by Western Australia\'s mining heritage'],
                    ['aspect' => 'Beach Lifestyle', 'description' => 'Pristine coastline promotes relaxed, sun-smart fashion choices']
                ],
                'color_palette' => [
                    ['name' => 'Desert Red', 'hex' => '#CD853F'],
                    ['name' => 'Ocean Teal', 'hex' => '#008080'],
                    ['name' => 'Sunset Purple', 'hex' => '#9B59B6'],
                    ['name' => 'Sand Stone', 'hex' => '#DEB887']
                ],
                'shopping_areas' => [
                    ['name' => 'Fremantle', 'description' => 'Alternative fashion and independent designers'],
                    ['name' => 'Northbridge', 'description' => 'Urban streetwear and creative boutiques'],
                    ['name' => 'Cottesloe', 'description' => 'Beach lifestyle and activewear specialists']
                ],
                'weather' => [
                    'avg_temp' => '16-30°C',
                    'rainfall' => 'Dry summers, wet winters',
                    'best_months' => 'Mar-Jun, Sep-Nov',
                    'style_note' => 'Mediterranean climate allows versatile styling with strong UV protection needs.'
                ],
                'style_gallery' => [
                    ['image' => 'gallery/perth-1.jpg', 'description' => 'Independent west coast', 'style_type' => 'Unique'],
                    ['image' => 'gallery/perth-2.jpg', 'description' => 'Desert landscape inspired', 'style_type' => 'Bold'],
                    ['image' => 'gallery/perth-3.jpg', 'description' => 'Coastal casual', 'style_type' => 'Relaxed'],
                    ['image' => 'gallery/perth-4.jpg', 'description' => 'Sun-smart streetwear', 'style_type' => 'Practical']
                ]
            ],
            'adelaide' => [
                'name' => 'Adelaide',
                'tagline' => 'Festival Fashion',
                'description' => 'Creative, cultural, and community-focused style inspired by Adelaide\'s arts scene',
                'region' => 'South Australia',
                'population' => '1.4M',
                'fashion_score' => 8,
                'streetwear_popularity' => 70,
                'meta_title' => 'Adelaide Streetwear Style Guide | Festival Fashion Tips',
                'meta_description' => 'Explore Adelaide\'s creative streetwear scene. Festival-inspired fashion, arts culture style, and community-focused fashion choices.',
                'keywords' => 'adelaide streetwear, adelaide fashion, festival fashion, arts culture style, adelaide style guide',
                'climate' => 'Mediterranean - transitional pieces',
                'vibe' => 'Creative, Cultural, Community-focused',
                'hero_image' => 'style-guides/adelaide-hero.jpg',
                'content' => $this->getAdelaideContent(),
                'featured_products' => Product::where('is_active', true)->limit(6)->get(),
                'styling_tips' => [
                    'Festival-ready pieces that work year-round',
                    'Support local and sustainable fashion choices',
                    'Creative expression through unique styling',
                    'Comfortable pieces for Adelaide\'s walkable city center'
                ],
                'style_characteristics' => [
                    ['title' => 'Festival Ready', 'description' => 'Comfortable, expressive pieces perfect for Adelaide\'s many festivals.'],
                    ['title' => 'Community Focused', 'description' => 'Supporting local designers and sustainable fashion choices.'],
                    ['title' => 'Cultural Celebration', 'description' => 'Fashion that honors Adelaide\'s diverse cultural heritage.']
                ],
                'key_pieces' => ['Festival Tee', 'Comfortable Denim', 'Artistic Hoodie', 'Walking Sneakers', 'Cultural Accessories', 'Sustainable Pieces'],
                'seasonal_styling' => [
                    'summer' => ['season' => 'Summer', 'icon' => 'sun', 'description' => 'Festival season styling', 'tips' => ['Breathable festival wear', 'Sun protection for outdoor events', 'Comfortable all-day pieces']],
                    'autumn' => ['season' => 'Autumn', 'icon' => 'leaf', 'description' => 'Arts season begins', 'tips' => ['Light layering for events', 'Creative expression pieces', 'Indoor-outdoor versatility']],
                    'winter' => ['season' => 'Winter', 'icon' => 'cloud', 'description' => 'Cozy cultural season', 'tips' => ['Warm layers for events', 'Cultural celebration wear', 'Community gathering comfort']],
                    'spring' => ['season' => 'Spring', 'icon' => 'seedling', 'description' => 'Festival preparation', 'tips' => ['Transitional weather pieces', 'Event-ready comfort', 'Creative layering']]
                ],
                'cultural_influences' => [
                    ['aspect' => 'Festival Culture', 'description' => 'Year-round festivals influence comfortable, expressive fashion choices'],
                    ['aspect' => 'Arts Community', 'description' => 'Creative community supports unique, artistic fashion expression'],
                    ['aspect' => 'Multicultural Heritage', 'description' => 'Diverse cultural backgrounds create rich, inclusive fashion choices']
                ],
                'color_palette' => [
                    ['name' => 'Festival Purple', 'hex' => '#8E44AD'],
                    ['name' => 'Arts Gold', 'hex' => '#F39C12'],
                    ['name' => 'Community Green', 'hex' => '#27AE60'],
                    ['name' => 'Cultural Orange', 'hex' => '#E67E22']
                ],
                'shopping_areas' => [
                    ['name' => 'Adelaide Central Market', 'description' => 'Local designers and unique handmade pieces'],
                    ['name' => 'Rundle Mall', 'description' => 'Mainstream and emerging Australian brands'],
                    ['name' => 'Norwood', 'description' => 'Vintage and sustainable fashion boutiques']
                ],
                'weather' => [
                    'avg_temp' => '14-26°C',
                    'rainfall' => 'Moderate, winter focused',
                    'best_months' => 'Mar-Jun, Sep-Nov',
                    'style_note' => 'Mediterranean climate perfect for layered, festival-ready styling year-round.'
                ],
                'style_gallery' => [
                    ['image' => 'gallery/adelaide-1.jpg', 'description' => 'Festival fashion ready', 'style_type' => 'Expressive'],
                    ['image' => 'gallery/adelaide-2.jpg', 'description' => 'Arts community style', 'style_type' => 'Creative'],
                    ['image' => 'gallery/adelaide-3.jpg', 'description' => 'Cultural celebration', 'style_type' => 'Cultural'],
                    ['image' => 'gallery/adelaide-4.jpg', 'description' => 'Sustainable choices', 'style_type' => 'Conscious']
                ]
            ]
        ];

        $guide = $guides[$city] ?? abort(404);
        }

        // Get related cities (other cities excluding the current one)
        if (isset($dbGuide)) {
            $relatedCities = StyleGuide::where('slug', '!=', $city)
                ->where('is_active', true)
                ->take(3)
                ->get()
                ->map(function ($guide) {
                    return [
                        'name' => $guide->name,
                        'description' => $guide->description,
                        'image' => $guide->hero_image
                    ];
                })
                ->toArray();
        } else {
            $relatedCities = collect($guides)
                ->except($city)
                ->take(3)
                ->map(function ($guide, $citySlug) {
                    return [
                        'name' => $guide['name'],
                        'description' => $guide['description'],
                        'image' => $guide['hero_image']
                    ];
                })
                ->toArray();
        }

        return view('style-guide.show', compact('guide', 'city', 'relatedCities'));
    }

    private function getMelbourneContent()
    {
        return '<h2>Melbourne: Where Creativity Meets Street Style</h2>
        
        <p>Melbourne\'s fashion scene is as dynamic and unpredictable as its famous weather. The city\'s creative energy flows through its fashion choices, creating a unique style that celebrates artistic expression and individual creativity.</p>
        
        <h3>The Melbourne Aesthetic</h3>
        
        <div class="row">
            <div class="col-md-6">
                <h4>Layering Mastery</h4>
                <p>Melbourne\'s "four seasons in one day" weather demands layering skills. Think lightweight cardigans over tees, denim jackets over hoodies, and accessories that can be added or removed as needed.</p>
            </div>
            <div class="col-md-6">
                <h4>Artistic Expression</h4>
                <p>Melbourne streetwear embraces bold prints, unique silhouettes, and pieces that tell a story. It\'s not uncommon to see vintage band tees paired with contemporary tailoring.</p>
            </div>
        </div>
        
        <h3>Melbourne Street Style Essentials</h3>
        
        <ul>
            <li><strong>The Perfect Hoodie:</strong> Oversized but not sloppy, in neutral or artistic prints</li>
            <li><strong>Layering Pieces:</strong> Lightweight jackets, cardigans, and vests</li>
            <li><strong>Statement Accessories:</strong> Unique jewelry, artistic bags, creative footwear</li>
            <li><strong>All-Black Everything:</strong> Melbourne\'s unofficial uniform, always chic</li>
        </ul>
        
        <h3>Seasonal Styling Tips</h3>
        
        <h4>Spring (September-November)</h4>
        <p>Light layers and transitional pieces. Denim jackets, lightweight hoodies, and comfortable sneakers.</p>
        
        <h4>Summer (December-February)</h4>
        <p>Breathable fabrics and sun protection. Cotton tees, linen blends, and stylish hats.</p>
        
        <h4>Autumn (March-May)</h4>
        <p>Perfect hoodie weather. Embrace cozy textures and warm earth tones.</p>
        
        <h4>Winter (June-August)</h4>
        <p>Serious layering time. Heavy hoodies, thermal underlayers, and statement outerwear.</p>';
    }

    private function getSydneyContent()
    {
        return '<h2>Sydney: Effortless Coastal Sophistication</h2>
        
        <p>Sydney style embodies the city\'s laid-back yet sophisticated lifestyle. It\'s about looking effortlessly put-together whether you\'re catching waves at Bondi or attending a rooftop bar in the CBD.</p>
        
        <h3>The Sydney Vibe</h3>
        
        <div class="row">
            <div class="col-md-6">
                <h4>Beach to City</h4>
                <p>Sydney fashion is about versatility. Pieces need to work whether you\'re walking along the harbour or grabbing coffee in Surry Hills.</p>
            </div>
            <div class="col-md-6">
                <h4>Relaxed Sophistication</h4>
                <p>Think elevated basics, quality fabrics, and timeless silhouettes that never try too hard but always look right.</p>
            </div>
        </div>
        
        <h3>Sydney Streetwear Staples</h3>
        
        <ul>
            <li><strong>Quality Basics:</strong> Premium cotton tees and well-fitted hoodies</li>
            <li><strong>Denim Done Right:</strong> Classic cuts in quality washes</li>
            <li><strong>Versatile Footwear:</strong> Clean sneakers that work everywhere</li>
            <li><strong>Sun-Smart Accessories:</strong> Stylish sunglasses and caps</li>
        </ul>
        
        <h3>Neighborhood Style Guides</h3>
        
        <h4>Bondi Beach</h4>
        <p>Relaxed beach vibes with quality activewear and comfortable casualwear.</p>
        
        <h4>Surry Hills</h4>
        <p>Creative casual with an edge. Unique pieces mixed with classic staples.</p>
        
        <h4>CBD</h4>
        <p>Smart casual that can transition from work to play seamlessly.</p>
        
        <h4>Newtown</h4>
        <p>Alternative and artistic with vintage influences and creative expression.</p>';
    }

    private function getBrisbaneContent()
    {
        return '<h2>Brisbane: Tropical Streetwear Done Right</h2>
        
        <p>Brisbane\'s subtropical climate demands smart fabric choices and breathable styling. The city\'s outdoor lifestyle and vibrant culture influence fashion choices that prioritize comfort without sacrificing style.</p>
        
        <h3>Climate-Smart Fashion</h3>
        
        <div class="row">
            <div class="col-md-6">
                <h4>Fabric Focus</h4>
                <p>Cotton blends, linen, and moisture-wicking materials are essential. Avoid heavy fabrics that trap heat and humidity.</p>
            </div>
            <div class="col-md-6">
                <h4>Color Psychology</h4>
                <p>Light colors reflect heat while bold tropical hues embrace Brisbane\'s vibrant outdoor culture.</p>
            </div>
        </div>
        
        <h3>Brisbane Style Essentials</h3>
        
        <ul>
            <li><strong>Breathable Tees:</strong> Cotton and bamboo blends in light colors</li>
            <li><strong>Lightweight Hoodies:</strong> For air-conditioned spaces and cooler evenings</li>
            <li><strong>Comfortable Shorts:</strong> Quality cuts that look intentional, not sloppy</li>
            <li><strong>Sun Protection:</strong> UPF clothing, wide-brim hats, and UV sunglasses</li>
        </ul>
        
        <h3>Seasonal Considerations</h3>
        
        <h4>Wet Season (November-April)</h4>
        <p>Quick-dry fabrics and covered footwear. Lightweight rain protection that packs small.</p>
        
        <h4>Dry Season (May-October)</h4>
        <p>Perfect weather for layering. Light jackets for evening and air-conditioned spaces.</p>';
    }

    private function getPerthContent()
    {
        return '<h2>Perth: Independent West Coast Style</h2>
        
        <p>Perth\'s geographic isolation has fostered a unique, independent fashion sense. The city\'s stunning coastline and desert landscapes inspire bold, confident style choices.</p>
        
        <h3>The Perth Spirit</h3>
        
        <div class="row">
            <div class="col-md-6">
                <h4>Independent Minded</h4>
                <p>Perth fashion doesn\'t follow east coast trends blindly. It creates its own rules based on lifestyle and climate needs.</p>
            </div>
            <div class="col-md-6">
                <h4>Landscape Inspired</h4>
                <p>The contrast between pristine beaches and rugged interior influences bold color choices and natural textures.</p>
            </div>
        </div>
        
        <h3>Perth Streetwear Characteristics</h3>
        
        <ul>
            <li><strong>Bold Colors:</strong> Inspired by stunning sunsets and vibrant landscapes</li>
            <li><strong>Quality Comfort:</strong> Pieces that work for outdoor adventures and city life</li>
            <li><strong>Sun Smart:</strong> UV protection built into stylish designs</li>
            <li><strong>Unique Finds:</strong> Supporting local brands and independent designers</li>
        </ul>';
    }

    private function getAdelaideContent()
    {
        return '<h2>Adelaide: Festival City Fashion</h2>
        
        <p>Adelaide\'s rich arts and festival culture influences its fashion scene. The city celebrates creativity, community, and cultural expression through thoughtful style choices.</p>
        
        <h3>Festival-Inspired Style</h3>
        
        <div class="row">
            <div class="col-md-6">
                <h4>Creative Expression</h4>
                <p>Adelaide fashion celebrates individual creativity and artistic expression through unique styling and bold choices.</p>
            </div>
            <div class="col-md-6">
                <h4>Community Focus</h4>
                <p>Supporting local brands and sustainable choices reflects Adelaide\'s community-minded culture.</p>
            </div>
        </div>
        
        <h3>Adelaide Style Elements</h3>
        
        <ul>
            <li><strong>Festival Ready:</strong> Comfortable pieces that work for outdoor events</li>
            <li><strong>Creative Details:</strong> Unique prints, artistic graphics, and handmade elements</li>
            <li><strong>Sustainable Choices:</strong> Supporting local and eco-conscious brands</li>
            <li><strong>Cultural Celebration:</strong> Pieces that honor diverse cultural backgrounds</li>
        </ul>';
    }
}