<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Category;
use App\Models\CompanyInfo;
use Illuminate\Http\Request;

class TrackingController extends Controller
{
    /**
     * Display the tracking form.
     */
    public function index()
    {
        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();
        
        return view('tracking.index', compact('categories', 'companyInfo'));
    }

    /**
     * Track an order.
     */
    public function track(Request $request)
    {
        $validated = $request->validate([
            'tracking_number' => 'required_without:order_number|string',
            'order_number' => 'required_without:tracking_number|string',
            'email' => 'required|email'
        ]);

        // Find order by tracking number or order number
        $query = Order::with(['items.product', 'shippingAddress']);
        
        if ($request->filled('tracking_number')) {
            $query->where('tracking_number', $validated['tracking_number']);
        } else {
            $query->where('order_number', $validated['order_number']);
        }
        
        $order = $query->where('customer_email', $validated['email'])->first();

        if (!$order) {
            return back()->with('error', 'No order found with the provided information. Please check your details and try again.');
        }

        // Parse tracking updates
        $trackingUpdates = [];
        if ($order->tracking_updates) {
            $trackingUpdates = is_string($order->tracking_updates) 
                ? json_decode($order->tracking_updates, true) 
                : $order->tracking_updates;
        }

        // Get categories and company info for the layout
        $categories = Category::where('is_active', true)->get();
        $companyInfo = CompanyInfo::first();

        return view('tracking.show', compact('order', 'trackingUpdates', 'categories', 'companyInfo'));
    }
}