<?php
// app/Models/Banner.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Banner extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'subtitle',
        'description',
        'image',
        'images',
        'slide_duration',
        'animation_type',
        'auto_play',
        'button_text',
        'button_link',
        'position', // hero, promo_left, promo_right, home_hero_slider, below_categories_slider
        'sort_order',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'auto_play' => 'boolean',
        'sort_order' => 'integer',
        'slide_duration' => 'integer',
        'images' => 'array'
    ];

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeHero($query)
    {
        return $query->where('position', 'hero');
    }

    public function scopePromo($query)
    {
        return $query->whereIn('position', ['promo_left', 'promo_right']);
    }

    public function scopeHomeHeroSlider($query)
    {
        return $query->where('position', 'home_hero_slider');
    }

    public function scopeBelowCategoriesSlider($query)
    {
        return $query->where('position', 'below_categories_slider');
    }

    public function scopeTrainSlider($query)
    {
        return $query->where('position', 'train_slider');
    }

    /**
     * Get slide images array with ImageHelper URLs
     */
    public function getSlideImages()
    {
        if (!empty($this->images) && is_array($this->images)) {
            return array_map(function($image) {
                return \App\Helpers\ImageHelper::getImageUrl($image);
            }, $this->images);
        }

        // Fallback to single image
        if ($this->image) {
            return [\App\Helpers\ImageHelper::getImageUrl($this->image)];
        }

        return [];
    }

    /**
     * Get slide settings for JavaScript
     */
    public function getSlideSettings()
    {
        return [
            'duration' => $this->slide_duration * 1000, // Convert to milliseconds
            'animation' => $this->animation_type,
            'autoPlay' => $this->auto_play,
            'imageCount' => count($this->getSlideImages())
        ];
    }

    /**
     * Check if banner has multiple images (is a slideshow)
     */
    public function isSlideshow()
    {
        return !empty($this->images) && is_array($this->images) && count($this->images) > 1;
    }

    /**
     * Get primary image URL for admin display
     */
    public function getPrimaryImageUrl()
    {
        if (!empty($this->images) && is_array($this->images)) {
            return \App\Helpers\ImageHelper::getImageUrl($this->images[0]);
        }

        if ($this->image) {
            return \App\Helpers\ImageHelper::getImageUrl($this->image);
        }

        return null;
    }

    /**
     * Get button URL - ensure it starts with http or /
     */
    public function getButtonUrl()
    {
        $link = $this->button_link;

        if (empty($link)) {
            return route('shop.index');
        }

        // If it doesn't start with http or /, prepend /
        if (!str_starts_with($link, 'http') && !str_starts_with($link, '/')) {
            $link = '/' . $link;
        }

        return $link;
    }
}