<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Carbon\Carbon;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_number', 
        'user_id', 
        'customer_name', 
        'customer_email', 
        'customer_phone',
        'subtotal', 
        'tax_amount', 
        'shipping_amount', 
        'discount_amount',
        'total_amount',
        'status', 
        'payment_status', 
        'payment_method', 
        'transaction_id',
        'stripe_payment_intent',
        'notes', 
        'shipped_at', 
        'delivered_at',
        'paid_at',
        'tracking_number',
        'tracking_updates',
        'carrier',
        'estimated_delivery',
        'coupon_code',
        'shipping_method',
        'shipping_city',
        'shipping_state'
    ];

    protected $casts = [
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'shipped_at' => 'datetime',
        'delivered_at' => 'datetime',
        'paid_at' => 'datetime',
        'estimated_delivery' => 'date',
        'tracking_updates' => 'array'
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($order) {
            if (empty($order->order_number)) {
                $order->order_number = 'ORD-' . strtoupper(uniqid());
            }
        });
    }

    /**
     * Get the user that owns the order.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the order items for the order.
     */
    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the shipping address for the order.
     */
    public function shippingAddress()
    {
        return $this->hasOne(ShippingAddress::class);
    }

    /**
     * Get the status badge color attribute.
     */
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'processing' => 'info',
            'shipped' => 'primary',
            'out_for_delivery' => 'info',
            'delivered' => 'success',
            'cancelled' => 'danger',
            'refunded' => 'secondary'
        ];

        return $badges[$this->status] ?? 'secondary';
    }

    /**
     * Get the payment status badge color attribute.
     */
    public function getPaymentStatusBadgeAttribute()
    {
        $badges = [
            'pending' => 'warning',
            'paid' => 'success',
            'failed' => 'danger',
            'refunded' => 'secondary'
        ];

        return $badges[$this->payment_status] ?? 'secondary';
    }

    /**
     * Check if order can be cancelled.
     */
    public function canBeCancelled()
    {
        return in_array($this->status, ['pending', 'processing']) && 
               $this->payment_status !== 'refunded';
    }

    /**
     * Check if order can be refunded.
     */
    public function canBeRefunded()
    {
        return $this->payment_status === 'paid' && 
               !in_array($this->status, ['refunded', 'cancelled']);
    }

    /**
     * Add a tracking update to the order.
     */
    public function addTrackingUpdate($message, $status = 'pending')
    {
        $updates = $this->tracking_updates ?? [];
        
        $updates[] = [
            'message' => $message,
            'status' => $status,
            'timestamp' => now()->format('Y-m-d H:i:s'),
            'date' => now()->format('Y-m-d H:i:s'), // For compatibility
            'user' => auth()->user() ? auth()->user()->name : 'System'
        ];
        
        $this->tracking_updates = $updates;
        $this->save();
    }

    /**
     * Get formatted tracking updates.
     */
    public function getFormattedTrackingUpdates()
    {
        if (!$this->tracking_updates) {
            return [];
        }

        return collect($this->tracking_updates)->map(function ($update) {
            return [
                'message' => $update['message'] ?? '',
                'status' => $update['status'] ?? 'info',
                'timestamp' => isset($update['timestamp']) 
                    ? Carbon::parse($update['timestamp'])->format('M d, Y h:i A')
                    : (isset($update['date']) 
                        ? Carbon::parse($update['date'])->format('M d, Y h:i A')
                        : ''),
                'user' => $update['user'] ?? 'System'
            ];
        })->toArray();
    }

    /**
     * Get tracking status badge color.
     */
    public function getTrackingStatusBadge($status)
    {
        $badges = [
            'pending' => 'secondary',
            'processing' => 'info',
            'shipped' => 'primary',
            'in_transit' => 'info',
            'out_for_delivery' => 'warning',
            'delivered' => 'success',
            'cancelled' => 'danger',
            'returned' => 'dark'
        ];

        return $badges[$status] ?? 'secondary';
    }

    /**
     * Scope for filtering orders by status.
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for filtering orders by payment status.
     */
    public function scopePaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    /**
     * Scope for filtering orders by date range.
     */
    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('created_at', [$from, $to]);
    }

    /**
     * Calculate order totals.
     */
    public function calculateTotals($city = null, $state = null)
    {
        $subtotal = $this->items->sum(function ($item) {
            return $item->price * $item->quantity;
        });

        $this->subtotal = $subtotal;
        
        // Calculate tax based on admin-configured tax rates
        $taxAmount = 0.0;
        if (class_exists(\App\Models\TaxRate::class)) {
            $taxRate = null;
            if ($city && $state) {
                $taxRate = \App\Models\TaxRate::findByCity($city, $state);
            } elseif ($this->shippingAddress) {
                $taxRate = \App\Models\TaxRate::findByCity(
                    $this->shippingAddress->city,
                    $this->shippingAddress->state
                );
            }
            
            if ($taxRate && method_exists($taxRate, 'calculateTax')) {
                $taxAmount = $taxRate->calculateTax($subtotal);
            } else {
                // Fallback to 10% if no tax rate found
                $taxAmount = $subtotal * 0.1;
            }
        } else {
            // Fallback to 10% if TaxRate model doesn't exist
            $taxAmount = $subtotal * 0.1;
        }
        
        $this->tax_amount = $taxAmount;
        $this->total_amount = $subtotal + $this->tax_amount + $this->shipping_amount - $this->discount_amount;
        
        return $this;
    }
}