<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ShippingAddress extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'order_id', 
        'first_name', 
        'last_name', 
        'company',
        'phone',
        'address_line_1', 
        'address_line_2', 
        'city', 
        'state',
        'postcode',
        'postal_code',
        'country',
        'is_default'
    ];

    protected $casts = [
        'is_default' => 'boolean'
    ];

    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function getFullNameAttribute()
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    public function getFullAddressAttribute()
    {
        $parts = [];
        if ($this->address_line_1) $parts[] = $this->address_line_1;
        if ($this->address_line_2) $parts[] = $this->address_line_2;
        
        $cityStatePost = trim("{$this->city} {$this->state} " . ($this->postcode ?? $this->postal_code));
        if ($cityStatePost) $parts[] = $cityStatePost;
        if ($this->country) $parts[] = $this->country;

        return implode(', ', $parts);
    }

    /**
     * Get formatted address for display
     */
    public function getFormattedAddressAttribute()
    {
        $address = $this->address_line_1;
        
        if ($this->address_line_2) {
            $address .= "\n" . $this->address_line_2;
        }
        
        $address .= "\n" . $this->city . ', ' . $this->state . ' ' . ($this->postcode ?? $this->postal_code);
        $address .= "\n" . $this->country;
        
        return $address;
    }

    /**
     * Scope to get default addresses
     */
    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /**
     * Set as default address for user
     */
    public function setAsDefault()
    {
        // Remove default from other addresses
        self::where('user_id', $this->user_id)
            ->where('id', '!=', $this->id)
            ->update(['is_default' => false]);
        
        // Set this as default
        $this->is_default = true;
        $this->save();
    }
}