<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Helpers\ImageHelper;

class StyleGuide extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'tagline',
        'description',
        'region',
        'population',
        'fashion_score',
        'streetwear_popularity',
        'meta_title',
        'meta_description',
        'keywords',
        'climate',
        'vibe',
        'hero_image',
        'content',
        'styling_tips',
        'style_characteristics',
        'key_pieces',
        'seasonal_styling',
        'cultural_influences',
        'color_palette',
        'shopping_areas',
        'weather',
        'style_gallery',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'styling_tips' => 'array',
        'style_characteristics' => 'array',
        'key_pieces' => 'array',
        'seasonal_styling' => 'array',
        'cultural_influences' => 'array',
        'color_palette' => 'array',
        'shopping_areas' => 'array',
        'weather' => 'array',
        'style_gallery' => 'array',
        'is_active' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->slug)) {
                $model->slug = Str::slug($model->name);
            }
        });

        static::updating(function ($model) {
            if ($model->isDirty('name')) {
                $model->slug = Str::slug($model->name);
            }
        });
    }

    // Scope for active style guides
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope for ordering by sort_order
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    // Get hero image URL using ImageHelper
    public function getHeroImageUrlAttribute()
    {
        return ImageHelper::getImageUrl($this->hero_image);
    }

    // Get style gallery image URLs using ImageHelper
    public function getStyleGalleryUrlsAttribute()
    {
        if (empty($this->style_gallery)) {
            return [];
        }

        return collect($this->style_gallery)->map(function ($item) {
            if (is_array($item) && isset($item['image'])) {
                $item['image_url'] = ImageHelper::getImageUrl($item['image']);
                return $item;
            }
            return $item;
        })->toArray();
    }

    // Get route key name for route model binding
    public function getRouteKeyName()
    {
        return 'slug';
    }

    // Check if hero image exists
    public function hasHeroImage()
    {
        return !empty($this->hero_image) && ImageHelper::imageExists($this->hero_image);
    }
}
