<?php

namespace App\View\Components;

use Illuminate\View\Component;
use Illuminate\Support\Facades\Request;

class Breadcrumb extends Component
{
    public $breadcrumbs;
    public $jsonLd;
    
    /**
     * Create a new component instance.
     */
    public function __construct($breadcrumbs = null)
    {
        $this->breadcrumbs = $breadcrumbs ?? $this->generateBreadcrumbs();
        $this->jsonLd = $this->generateJsonLd();
    }
    
    /**
     * Auto-generate breadcrumbs based on current route
     */
    private function generateBreadcrumbs()
    {
        $breadcrumbs = [
            ['name' => 'Home', 'url' => route('home')]
        ];
        
        $routeName = Request::route()->getName();
        $segments = Request::segments();
        
        switch ($routeName) {
            case 'shop.index':
                $breadcrumbs[] = ['name' => 'Shop', 'url' => route('shop.index')];
                break;
                
            case 'shop.category':
                $breadcrumbs[] = ['name' => 'Shop', 'url' => route('shop.index')];
                $category = Request::route('category');
                if ($category) {
                    $breadcrumbs[] = ['name' => $category->name, 'url' => route('shop.category', $category)];
                }
                break;
                
            case 'shop.product':
                $breadcrumbs[] = ['name' => 'Shop', 'url' => route('shop.index')];
                $product = Request::route('product');
                if ($product && $product->category) {
                    $breadcrumbs[] = ['name' => $product->category->name, 'url' => route('shop.category', $product->category)];
                }
                if ($product) {
                    $breadcrumbs[] = ['name' => $product->name, 'url' => route('shop.product', $product)];
                }
                break;
                
            case 'blogs.index':
                $breadcrumbs[] = ['name' => 'Blog', 'url' => route('blogs.index')];
                break;
                
            case 'blogs.show':
                $breadcrumbs[] = ['name' => 'Blog', 'url' => route('blogs.index')];
                $blog = Request::route('slug');
                if ($blog) {
                    $breadcrumbs[] = ['name' => $this->truncateTitle($blog), 'url' => Request::url()];
                }
                break;
                
            case 'about':
                $breadcrumbs[] = ['name' => 'About Us', 'url' => route('about')];
                break;
                
            case 'contact':
                $breadcrumbs[] = ['name' => 'Contact', 'url' => route('contact')];
                break;
        }
        
        return $breadcrumbs;
    }
    
    /**
     * Truncate long titles for breadcrumbs
     */
    private function truncateTitle($title, $length = 30)
    {
        if (is_string($title)) {
            return strlen($title) > $length ? substr($title, 0, $length) . '...' : $title;
        }
        return $title;
    }
    
    /**
     * Generate JSON-LD structured data for breadcrumbs
     */
    private function generateJsonLd()
    {
        $itemList = [];
        
        foreach ($this->breadcrumbs as $index => $breadcrumb) {
            $itemList[] = [
                "@type" => "ListItem",
                "position" => $index + 1,
                "name" => $breadcrumb['name'],
                "item" => $breadcrumb['url']
            ];
        }
        
        return [
            "@context" => "https://schema.org",
            "@type" => "BreadcrumbList",
            "itemListElement" => $itemList
        ];
    }
    
    /**
     * Get the view / contents that represent the component.
     */
    public function render()
    {
        return view('components.breadcrumb');
    }
}