<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Category;
use App\Models\Product;
use App\Models\CompanyInfo;
use App\Models\Testimonial;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        // 1) Admin user (idempotent)
        User::updateOrCreate(
            ['email' => 'admin@example.com'],
            [
                'name' => 'Admin',
                'password' => Hash::make('password'),
                'email_verified_at' => now(),
            ]
        );

        // 2) Company info (idempotent)
        CompanyInfo::updateOrCreate(
            ['company_name' => 'Global Export Co.'],
            [
                'about_us' => 'We are a leading manufacturer and exporter of quality products worldwide. With over 15 years of experience in international trade, we have established ourselves as a trusted partner for businesses across the globe.',
                'mission' => 'To provide high-quality products to global markets with exceptional service and competitive pricing.',
                'vision' => 'To be the preferred export partner worldwide, known for reliability, quality, and innovation.',
                'phone' => '+1 234 567 890',
                'email' => 'info@globalexport.com',
                'address' => '123 Business Street, Industrial Area, City, Country 12345',
                'years_experience' => 15,
                'happy_clients' => 500,
                'products_count' => 100,
                'export_countries' => 'USA, UK, Germany, France, UAE, Saudi Arabia, Australia, Canada, Japan, Singapore',
                'whatsapp' => '+1234567890',
                // If CompanyInfo 'social_links' column is JSON/castable, you may pass array.
                // If it's a plain string column, adapt accordingly (e.g. json_encode([...])).
                'social_links' => [
                    'facebook' => 'https://facebook.com/yourcompany',
                    'twitter' => 'https://twitter.com/yourcompany',
                    'linkedin' => 'https://linkedin.com/company/yourcompany',
                    'instagram' => 'https://instagram.com/yourcompany'
                ],
            ]
        );

        // 3) Categories (idempotent)
        $categories = [
            'Electronics' => 'High-quality electronic products and components for global markets',
            'Textiles' => 'Premium textile and fabric products for fashion and home furnishing',
            'Machinery' => 'Industrial machinery and equipment for various applications',
            'Food Products' => 'Quality food and agricultural products meeting international standards',
            'Chemicals' => 'Industrial and commercial chemicals for diverse industries',
            'Furniture' => 'Modern and traditional furniture for homes and offices'
        ];

        foreach ($categories as $name => $description) {
            Category::updateOrCreate(
                ['slug' => Str::slug($name)],
                [
                    'name' => $name,
                    'description' => $description,
                    'is_active' => true,
                    'sort_order' => 0,
                ]
            );
        }

        // 4) Sample products (idempotent)
        $electronicsCategory = Category::where('slug', 'electronics')->first();
        if ($electronicsCategory) {
            Product::updateOrCreate(
                ['slug' => 'smart-led-tv-55'],
                [
                    'category_id' => $electronicsCategory->id,
                    'name' => 'Smart LED TV 55"',
                    'description' => 'High-quality Smart LED TV with 4K resolution, HDR support, and smart features. Perfect for home entertainment.',
                    'specifications' => "Screen Size: 55 inches\nResolution: 4K Ultra HD (3840x2160)\nHDR: HDR10, Dolby Vision\nSmart TV: Yes\nConnectivity: WiFi, Bluetooth, HDMI x3, USB x2",
                    'price' => 599.99,
                    'sku' => 'TV-LED-55-001',
                    'featured_image' => 'products/sample.jpg',
                    'is_featured' => true,
                    'is_active' => true,
                    'meta_title' => 'Smart LED TV 55\" - Best Price for Export',
                    'meta_description' => 'High-quality 55 inch Smart LED TV with 4K resolution for export. Competitive wholesale prices.',
                    'meta_keywords' => 'smart tv, led tv, 4k tv, export, wholesale'
                ]
            );

            Product::updateOrCreate(
                ['slug' => 'wireless-bluetooth-headphones'],
                [
                    'category_id' => $electronicsCategory->id,
                    'name' => 'Wireless Bluetooth Headphones',
                    'description' => 'Premium wireless headphones with active noise cancellation and superior sound quality.',
                    'specifications' => "Type: Over-ear\nConnectivity: Bluetooth 5.0\nBattery Life: 30 hours\nNoise Cancellation: Active\nCharging: USB-C fast charging",
                    'price' => 89.99,
                    'sku' => 'HP-BT-001',
                    'featured_image' => 'products/sample.jpg',
                    'is_featured' => true,
                    'is_active' => true
                ]
            );
        }

        // 5) Testimonials (idempotent)
        Testimonial::updateOrCreate(
            ['client_name' => 'John Smith', 'client_company' => 'Tech Imports LLC'],
            [
                'client_position' => 'Purchase Manager',
                'client_country' => 'United States',
                'content' => 'Excellent quality products and reliable delivery. We have been working with them for 3 years and are very satisfied with their service.',
                'rating' => 5,
                'is_active' => true
            ]
        );

        Testimonial::updateOrCreate(
            ['client_name' => 'Maria Garcia', 'client_company' => 'European Traders'],
            [
                'client_position' => 'CEO',
                'client_country' => 'Spain',
                'content' => 'Professional team and competitive prices. They always deliver on time and maintain consistent quality.',
                'rating' => 5,
                'is_active' => true
            ]
        );

        Testimonial::updateOrCreate(
            ['client_name' => 'Ahmed Hassan', 'client_company' => 'Middle East Imports'],
            [
                'client_position' => 'Procurement Director',
                'client_country' => 'UAE',
                'content' => 'Great communication and excellent product quality. Highly recommended for international trade.',
                'rating' => 5,
                'is_active' => true
            ]
        );

        // 6) Optional: call AdminUserSeeder if it exists (keeps compatibility with other projects)
        if (class_exists('Database\\Seeders\\AdminUserSeeder')) {
            $this->call(\Database\Seeders\AdminUserSeeder::class);
        }

        // Add other seeder calls here if you create them later:
        // $this->call([
        //     AnotherSeeder::class,
        // ]);
    }
}