@extends('layouts.app')

@section('title', 'My Orders')
@section('meta_description', 'View and manage all your orders')

@section('content')
<!-- Premium Account Hero -->
<section class="account-hero-premium">
    <div class="hero-overlay"></div>
    <div class="container">
        <div class="hero-content-account" data-aos="fade-up">
            <h1 class="hero-title-account">MY ORDERS</h1>
            <p class="hero-subtitle-account">Track and manage your purchases</p>
        </div>
    </div>
</section>

<!-- Orders Content -->
<section class="account-section-premium">
    <div class="container">
        <div class="row">
            <!-- Premium Sidebar -->
            <div class="col-lg-3" data-aos="fade-right">
                @include('account.sidebar')
            </div>

            <!-- Orders List -->
            <div class="col-lg-9">
                <!-- Orders Header -->
                <div class="orders-header-premium" data-aos="fade-up">
                    <h2 class="section-title-account">Order History</h2>
                    <div class="orders-filter">
                        <button class="filter-btn active" data-filter="all">
                            All Orders
                            <span class="filter-count">{{ $orders->total() }}</span>
                        </button>
                        <button class="filter-btn" data-filter="pending">
                            Pending
                            <span class="filter-count">{{ $orders->where('status', 'pending')->count() }}</span>
                        </button>
                        <button class="filter-btn" data-filter="delivered">
                            Delivered
                            <span class="filter-count">{{ $orders->where('status', 'delivered')->count() }}</span>
                        </button>
                    </div>
                </div>

                @if($orders->count() > 0)
                <!-- Orders List -->
                <div class="orders-list-premium">
                    @foreach($orders as $index => $order)
                    <div class="order-card-premium" data-aos="fade-up" data-aos-delay="{{ $index * 50 }}">
                        <div class="order-header">
                            <div class="order-info">
                                <h4 class="order-number">
                                    Order #{{ $order->order_number }}
                                </h4>
                                <p class="order-date">
                                    <i class="far fa-calendar"></i>
                                    {{ $order->created_at->format('F d, Y') }}
                                </p>
                            </div>
                            <div class="order-status-wrapper">
                                <span class="order-status-badge {{ $order->status }}">
                                    @if($order->status == 'delivered')
                                        <i class="fas fa-check-circle"></i>
                                    @elseif($order->status == 'shipped')
                                        <i class="fas fa-truck"></i>
                                    @elseif($order->status == 'processing')
                                        <i class="fas fa-box"></i>
                                    @elseif($order->status == 'cancelled')
                                        <i class="fas fa-times-circle"></i>
                                    @else
                                        <i class="fas fa-clock"></i>
                                    @endif
                                    {{ ucfirst(str_replace('_', ' ', $order->status)) }}
                                </span>
                            </div>
                        </div>
                        
                        <div class="order-body">
                            <div class="order-items-preview">
                                @foreach($order->items->take(3) as $item)
                                <div class="item-preview">
                                    @if($item->product_image)
                                    <img src="{{ asset('storage/' . $item->product_image) }}" 
                                         alt="{{ $item->product_name }}"
                                         class="item-thumb">
                                    @else
                                    <div class="item-thumb-placeholder">
                                        <i class="fas fa-image"></i>
                                    </div>
                                    @endif
                                </div>
                                @endforeach
                                @if($order->items->count() > 3)
                                <div class="item-preview more-items">
                                    <span>+{{ $order->items->count() - 3 }}</span>
                                </div>
                                @endif
                            </div>
                            
                            <div class="order-summary">
                                <div class="summary-item">
                                    <span class="summary-label">Items</span>
                                    <span class="summary-value">{{ $order->items->count() }}</span>
                                </div>
                                <div class="summary-item">
                                    <span class="summary-label">Tax Rate</span>
                                    <span class="summary-value">
                                        @if($order->taxRateInfo)
                                            @if($order->taxRateInfo->rate == 0)
                                                <span class="tax-free-badge">{{ $order->taxRateInfo->tax_name }} 0%</span>
                                            @else
                                                {{ $order->taxRateInfo->tax_name }} {{ $order->taxRateInfo->rate }}%
                                            @endif
                                        @else
                                            GST 10%
                                        @endif
                                    </span>
                                </div>
                                <div class="summary-item">
                                    <span class="summary-label">Shipping</span>
                                    <span class="summary-value">
                                        @if($order->shipping_method)
                                            {{ ucfirst($order->shipping_method) }}
                                            @if($order->shipping_amount == 0)
                                                <span class="free-shipping">FREE</span>
                                            @else
                                                ${{ number_format($order->shipping_amount, 2) }}
                                            @endif
                                        @else
                                            @if($order->shipping_amount == 0)
                                                <span class="free-shipping">FREE</span>
                                            @else
                                                ${{ number_format($order->shipping_amount, 2) }}
                                            @endif
                                        @endif
                                    </span>
                                </div>
                                <div class="summary-item">
                                    <span class="summary-label">Total</span>
                                    <span class="summary-value price">
                                        @if(isset($order->calculated_total_amount))
                                            ${{ number_format($order->calculated_total_amount, 2) }}
                                        @else
                                            ${{ number_format($order->total_amount, 2) }}
                                        @endif
                                    </span>
                                </div>
                                <div class="summary-item">
                                    <span class="summary-label">Payment</span>
                                    <span class="payment-status {{ $order->payment_status }}">
                                        {{ ucfirst($order->payment_status) }}
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="order-footer">
                            <div class="order-actions">
                                <a href="{{ route('account.order.detail', $order->order_number) }}" 
                                   class="btn-order-action primary">
                                    <i class="fas fa-eye"></i>
                                    View Details
                                </a>
                                
                                @if($order->tracking_number)
                                <a href="{{ route('order.track.index') }}?tracking_number={{ $order->tracking_number }}" 
                                   class="btn-order-action secondary">
                                    <i class="fas fa-truck"></i>
                                    Track Order
                                </a>
                                @endif
                                
                                <a href="{{ route('order.invoice', $order->order_number) }}" 
                                   class="btn-order-action outline" target="_blank">
                                    <i class="fas fa-download"></i>
                                    Invoice
                                </a>
                            </div>
                            
                            @if($order->estimated_delivery && $order->status == 'shipped')
                            <div class="delivery-info">
                                <i class="fas fa-shipping-fast"></i>
                                Est. Delivery: {{ \Carbon\Carbon::parse($order->estimated_delivery)->format('M d, Y') }}
                            </div>
                            @endif
                        </div>
                    </div>
                    @endforeach
                </div>
                
                <!-- Pagination -->
                <div class="pagination-wrapper" data-aos="fade-up">
                    {{ $orders->links('pagination::bootstrap-5') }}
                </div>
                @else
                <!-- Empty State -->
                <div class="empty-orders-premium" data-aos="zoom-in">
                    <div class="empty-icon">
                        <i class="fas fa-shopping-bag"></i>
                    </div>
                    <h3>No Orders Yet</h3>
                    <p>Looks like you haven't placed any orders yet</p>
                    <a href="{{ route('shop.index') }}" class="btn-start-shopping-premium">
                        START SHOPPING
                        <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
                @endif
            </div>
        </div>
    </div>
</section>

@push('styles')
<style>
/* Account Hero (reused from dashboard) */
.account-hero-premium {
    height: 30vh;
    min-height: 250px;
    background: linear-gradient(135deg, #000 0%, #333 100%);
    position: relative;
    display: flex;
    align-items: center;
    overflow: hidden;
}

.account-hero-premium::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: repeating-linear-gradient(
        45deg,
        transparent,
        transparent 35px,
        rgba(255,255,255,.02) 35px,
        rgba(255,255,255,.02) 70px
    );
}

.hero-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: radial-gradient(circle at center, transparent 0%, rgba(0,0,0,0.4) 100%);
}

.hero-content-account {
    text-align: center;
    color: white;
    position: relative;
    z-index: 2;
}

.hero-title-account {
    font-size: clamp(2rem, 4vw, 3rem);
    font-weight: 900;
    letter-spacing: 0.1em;
    margin-bottom: 0.5rem;
    font-family: 'Space Grotesk', sans-serif;
}

.hero-subtitle-account {
    font-size: 1.125rem;
    opacity: 0.9;
}

/* Account Section */
.account-section-premium {
    padding: 4rem 0;
    background: #fafafa;
    min-height: 60vh;
}

/* Orders Header */
.orders-header-premium {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    flex-wrap: wrap;
    gap: 1rem;
}

.section-title-account {
    font-size: 1.75rem;
    font-weight: 800;
    color: #000;
    margin: 0;
    font-family: 'Space Grotesk', sans-serif;
}

.orders-filter {
    display: flex;
    gap: 0.5rem;
}

.filter-btn {
    padding: 0.5rem 1rem;
    background: white;
    border: 1px solid #e0e0e0;
    color: #666;
    font-size: 0.875rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-btn:hover {
    background: #f8f8f8;
    border-color: #000;
    color: #000;
}

.filter-btn.active {
    background: #000;
    color: white;
    border-color: #000;
}

.filter-count {
    background: rgba(255,255,255,0.2);
    padding: 0.125rem 0.5rem;
    border-radius: 10px;
    font-size: 0.75rem;
}

.filter-btn.active .filter-count {
    background: rgba(255,255,255,0.2);
}

/* Order Cards */
.orders-list-premium {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
    margin-bottom: 3rem;
}

.order-card-premium {
    background: white;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
    overflow: hidden;
    transition: all 0.3s;
}

.order-card-premium:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 40px rgba(0,0,0,0.1);
}

.order-header {
    padding: 1.5rem;
    background: linear-gradient(135deg, #fafafa 0%, #fff 100%);
    border-bottom: 1px solid #f0f0f0;
    display: flex;
    justify-content: space-between;
    align-items: start;
}

.order-number {
    font-size: 1.125rem;
    font-weight: 700;
    color: #000;
    margin-bottom: 0.5rem;
}

.order-date {
    color: #666;
    font-size: 0.875rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.order-status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
    font-weight: 600;
    border-radius: 20px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.order-status-badge.pending,
.order-status-badge.processing {
    background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
    color: #856404;
}

.order-status-badge.shipped,
.order-status-badge.out_for_delivery {
    background: linear-gradient(135deg, #cce5ff 0%, #b8daff 100%);
    color: #004085;
}

.order-status-badge.delivered {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
}

.order-status-badge.cancelled {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
}

.order-body {
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 2rem;
}

.order-items-preview {
    display: flex;
    gap: 0.5rem;
}

.item-preview {
    position: relative;
}

.item-thumb,
.item-thumb-placeholder {
    width: 60px;
    height: 60px;
    border: 1px solid #e0e0e0;
    object-fit: cover;
}

.item-thumb-placeholder {
    display: flex;
    align-items: center;
    justify-content: center;
    background: #f8f8f8;
    color: #ccc;
}

.more-items {
    width: 60px;
    height: 60px;
    background: #000;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 600;
}

.order-summary {
    display: flex;
    gap: 3rem;
}

.summary-item {
    display: flex;
    flex-direction: column;
}

.summary-label {
    font-size: 0.75rem;
    color: #999;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    margin-bottom: 0.25rem;
}

.summary-value {
    font-size: 1rem;
    font-weight: 600;
    color: #000;
}

.summary-value.price {
    font-size: 1.25rem;
    color: #000;
}

.payment-status {
    padding: 0.25rem 0.75rem;
    border-radius: 12px;
    font-size: 0.875rem;
    font-weight: 600;
}

.payment-status.paid {
    background: #d4edda;
    color: #155724;
}

.payment-status.pending {
    background: #fff3cd;
    color: #856404;
}

.tax-free-badge {
    color: #28a745;
    font-weight: 700;
}

.free-shipping {
    color: #28a745;
    font-weight: 600;
    font-size: 0.75rem;
}

.order-footer {
    padding: 1.5rem;
    background: #fafafa;
    border-top: 1px solid #f0f0f0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.order-actions {
    display: flex;
    gap: 0.75rem;
}

.btn-order-action {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    font-size: 0.875rem;
    font-weight: 600;
    text-decoration: none;
    transition: all 0.3s;
    border: none;
}

.btn-order-action.primary {
    background: #000;
    color: white;
}

.btn-order-action.primary:hover {
    background: #333;
    transform: translateY(-2px);
}

.btn-order-action.secondary {
    background: white;
    color: #000;
    border: 1px solid #000;
}

.btn-order-action.secondary:hover {
    background: #000;
    color: white;
}

.btn-order-action.outline {
    background: transparent;
    color: #666;
    border: 1px solid #e0e0e0;
}

.btn-order-action.outline:hover {
    border-color: #000;
    color: #000;
}

.delivery-info {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #666;
    font-size: 0.875rem;
}

/* Empty State */
.empty-orders-premium {
    background: white;
    padding: 5rem 2rem;
    text-align: center;
    box-shadow: 0 10px 30px rgba(0,0,0,0.05);
}

.empty-icon {
    width: 100px;
    height: 100px;
    margin: 0 auto 2rem;
    background: linear-gradient(135deg, #f0f0f0 0%, #e0e0e0 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 3rem;
    color: #999;
    animation: float 3s ease-in-out infinite;
}

@keyframes float {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-10px); }
}

.empty-orders-premium h3 {
    font-size: 1.75rem;
    font-weight: 700;
    color: #333;
    margin-bottom: 0.5rem;
}

.empty-orders-premium p {
    color: #666;
    margin-bottom: 2rem;
}

.btn-start-shopping-premium {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem 2.5rem;
    background: #000;
    color: white;
    text-decoration: none;
    font-weight: 700;
    letter-spacing: 0.05em;
    transition: all 0.3s;
}

.btn-start-shopping-premium:hover {
    background: #333;
    transform: translateY(-2px);
    color: white;
}

/* Pagination */
.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 3rem;
}

.pagination {
    gap: 0.5rem;
}

.page-link {
    background: white;
    border: 1px solid #e0e0e0;
    color: #333;
    padding: 0.5rem 1rem;
    transition: all 0.3s;
}

.page-link:hover {
    background: #000;
    border-color: #000;
    color: white;
}

.page-item.active .page-link {
    background: #000;
    border-color: #000;
    color: white;
}

/* Responsive */
@media (max-width: 991px) {
    .order-body {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .order-summary {
        width: 100%;
        justify-content: space-between;
    }
    
    .order-footer {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
}

@media (max-width: 576px) {
    .orders-header-premium {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .orders-filter {
        width: 100%;
        overflow-x: auto;
    }
    
    .order-actions {
        flex-wrap: wrap;
        width: 100%;
    }
    
    .btn-order-action {
        flex: 1;
        justify-content: center;
    }
}
</style>
@endpush

@push('scripts')
<script>
// Filter functionality
document.querySelectorAll('.filter-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        // Remove active class from all buttons
        document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
        // Add active class to clicked button
        this.classList.add('active');
        
        // Filter logic would go here if implementing client-side filtering
        const filter = this.dataset.filter;
        console.log('Filter by:', filter);
    });
});

// Add hover animation to order cards
document.querySelectorAll('.order-card-premium').forEach(card => {
    card.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-5px) scale(1.01)';
    });
    
    card.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0) scale(1)';
    });
});
</script>
@endpush
@endsection